/*
 * Copyright 2017 George Aristy.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.llorllale.youtrack.api;

import org.llorllale.youtrack.api.session.Session;
import org.llorllale.youtrack.api.session.UnauthorizedException;

import java.io.IOException;
import java.time.Instant;
import java.util.List;
import java.util.Map;

/**
 * A {@link YouTrack} issue.
 * 
 * @author George Aristy (george.aristy@gmail.com)
 * @since 0.1.0
 */
public interface Issue {
  /**
   * The {@link Project} that the issue was created in.
   * 
   * @return the ID of the project that the issue was created in
   * @since 0.4.0
   */
  public Project project();

  /**
   * The issue's id.
   * 
   * @return the issue's id
   * @since 0.1.0
   */
  public String id();

  /**
   * The date the issue was created.
   * 
   * @return The date the issue was created.
   * @since 0.1.0
   */
  public Instant creationDate();

  /**
   * The issue's summary.
   * 
   * @return The issue's summary.
   * @since 0.1.0
   */
  public String summary();

  /**
   * The issue's description.
   * 
   * @return The issue's description.
   * @since 0.1.0
   */
  public String description();

  /**
   * Access to the issue's {@link User users}.
   * 
   * @return access to the issue's {@link User users}
   * @since 0.5.0
   */
  public UsersOfIssue users();

  /**
   * Access to the issue's {@link Comment comments}.
   * 
   * @return Access to the issue's {@link Comment comments}.
   * @since 0.4.0
   */
  public Comments comments();

  /**
   * Access to the issue's {@link TimeTrackEntry timetracking}.
   * 
   * <p><strong>Note:</strong> it is important that you first check whether timetracking is 
   * {@link TimeTracking#enabled() enabled} for the {@link Project}.</p>
   * 
   * @return access to the issue's {@link TimeTrackEntry timetracking}.
   * @since 0.4.0
   * @see TimeTracking#enabled() 
   */
  public IssueTimeTracking timetracking();

  /**
   * Returns the same {@link Issue} after refreshing its data from the server.
   * 
   * <p>
   * This is a convenient shortcut for {@code issue.project().issues().get(issue.id()).get()}.
   * </p>
   * 
   * @return the same {@link Issue} after refreshing its data from the server
   * @throws IOException if the server is unavailable
   * @throws UnauthorizedException if the user's {@link Session} is not authorized to perform this
   *     operation
   * @since 0.5.0
   */
  public Issue refresh() throws IOException, UnauthorizedException;

  /**
   * Updates this issue's {@link Field field} to the given {@link FieldValue value} and returns a 
   * new {@link Issue} reflecting those changes.
   * 
   * @param field the issue's {@link Field field}
   * @param value the field's {@link FieldValue value}
   * @return a new {@link Issue} reflecting the changes brought about by updating this issue with
   *     {@code args}
   * @throws IOException if the server is unavailable
   * @throws UnauthorizedException if the user's {@link Session} is not authorized to perform this
   *     operation
   * @since 0.7.0
   * @see #fields() 
   */
  public Issue update(Field field, FieldValue value) throws IOException, UnauthorizedException;

  /**
   * Updates this issue with the {@code fields} provided, returning a new {@link Issue} reflecting
   * those changes.
   * 
   * @param fields the collection of fields and their values to update
   * @return a new {@link Issue} reflecting those changes
   * @throws IOException if the server is unavailable
   * @throws UnauthorizedException if the user's {@link Session} is not authorized to perform this
   *     operation
   * @since 0.8.0
   * @see #update(org.llorllale.youtrack.api.Field, org.llorllale.youtrack.api.FieldValue) 
   */
  public Issue update(Map<Field, FieldValue> fields) throws IOException, UnauthorizedException;

  /**
   * All {@link AssignedField fields} of this {@link Issue}.
   * 
   * @return all {@link AssignedField fields} of this {@link Issue}
   * @since 0.8.0
   * @see #update(org.llorllale.youtrack.api.Field, org.llorllale.youtrack.api.FieldValue) 
   */
  public List<AssignedField> fields();
}
