/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.geowave.datastore.kudu.cli;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveException;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecuteResultHandler;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.ExecuteResultHandler;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.codehaus.plexus.archiver.tar.TarGZipUnArchiver;
import org.codehaus.plexus.logging.console.ConsoleLogger;
import org.locationtech.geowave.datastore.kudu.cli.RunKuduLocalOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KuduLocal {
    private static final Logger LOGGER = LoggerFactory.getLogger(KuduLocal.class);
    private static final String KUDU_REPO_URL = "https://archive.cloudera.com/cdh5/ubuntu/xenial/amd64/cdh/pool/contrib/k/kudu/";
    private static final String KUDU_DEB_PACKAGE = "kudu_1.7.0+cdh5.16.1+0-1.cdh5.16.1.p0.3~xenial-cdh5.16.1_amd64.deb";
    private static final String KUDU_MASTER = "kudu-master";
    private static final String KUDU_TABLET = "kudu-tserver";
    private static final long STARTUP_DELAY_MS = 1500L;
    public static final File DEFAULT_DIR = new File("./target/temp");
    private final int numTablets;
    private final File kuduLocalDir;
    private final File kuduDBDir;
    private final List<ExecuteWatchdog> watchdogs;

    public KuduLocal(RunKuduLocalOptions opt) {
        this(opt.getDirectory(), opt.getTablets());
    }

    public KuduLocal(String localDir, int numTablets) {
        this.kuduLocalDir = localDir != null && !localDir.contentEquals("") ? new File(localDir) : new File(DEFAULT_DIR, "kudu");
        if (!this.kuduLocalDir.exists() && !this.kuduLocalDir.mkdirs()) {
            LOGGER.error("unable to create directory {}", (Object)this.kuduLocalDir.getAbsolutePath());
        } else if (!this.kuduLocalDir.isDirectory()) {
            LOGGER.error("{} exists but is not a directory", (Object)this.kuduLocalDir.getAbsolutePath());
        }
        this.kuduDBDir = new File(this.kuduLocalDir, "db");
        this.watchdogs = new ArrayList<ExecuteWatchdog>();
        this.numTablets = numTablets;
    }

    public boolean start() {
        if (!this.isInstalled()) {
            try {
                if (!this.install()) {
                    return false;
                }
            }
            catch (IOException | ArchiveException e) {
                LOGGER.error("Kudu installation error: {}", (Object)e.getMessage());
                return false;
            }
        }
        try {
            this.startKuduLocal();
        }
        catch (IOException | InterruptedException e) {
            LOGGER.error("Kudu start error: {}", (Object)e.getMessage());
            return false;
        }
        return true;
    }

    public boolean isRunning() {
        return this.watchdogs.stream().anyMatch(w -> w.isWatching());
    }

    public void stop() {
        for (ExecuteWatchdog w : this.watchdogs) {
            w.destroyProcess();
        }
        try {
            Thread.sleep(1500L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public void destroyDB() throws IOException {
        try {
            FileUtils.deleteDirectory((File)this.kuduDBDir);
        }
        catch (IOException e) {
            LOGGER.error("Could not destroy database files", (Throwable)e);
            throw e;
        }
    }

    private boolean isInstalled() {
        File kuduMasterBinary = new File(this.kuduLocalDir, KUDU_MASTER);
        File kuduTabletBinary = new File(this.kuduLocalDir, KUDU_TABLET);
        boolean okMaster = kuduMasterBinary.exists() && kuduMasterBinary.canExecute();
        boolean okTablet = kuduTabletBinary.exists() && kuduTabletBinary.canExecute();
        return okMaster && okTablet;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean install() throws IOException, ArchiveException {
        File[] fileArray;
        LOGGER.info("Installing {}", (Object)KUDU_DEB_PACKAGE);
        LOGGER.debug("downloading kudu debian package");
        File debPackageFile = new File(this.kuduLocalDir, KUDU_DEB_PACKAGE);
        if (!debPackageFile.exists()) {
            HttpURLConnection.setFollowRedirects(true);
            URL url = new URL("https://archive.cloudera.com/cdh5/ubuntu/xenial/amd64/cdh/pool/contrib/k/kudu/kudu_1.7.0+cdh5.16.1+0-1.cdh5.16.1.p0.3~xenial-cdh5.16.1_amd64.deb");
            fileArray = null;
            try (FileOutputStream fos = new FileOutputStream(debPackageFile);){
                IOUtils.copy((InputStream)url.openStream(), (OutputStream)fos);
                fos.flush();
            }
            catch (Throwable object) {
                fileArray = object;
                throw object;
            }
        }
        LOGGER.debug("extracting kudu debian package data contents");
        File debDataTarGz = new File(this.kuduLocalDir, "data.tar.gz");
        if (!debDataTarGz.exists()) {
            fileArray = null;
            try (FileInputStream fis = new FileInputStream(debPackageFile);
                 ArchiveInputStream debInputStream2 = new ArchiveStreamFactory().createArchiveInputStream("ar", (InputStream)fis);){
                ArchiveEntry entry = null;
                while ((entry = debInputStream2.getNextEntry()) != null) {
                    if (!debDataTarGz.getName().equals(entry.getName())) continue;
                    try (FileOutputStream fos = new FileOutputStream(debDataTarGz);){
                        IOUtils.copy((InputStream)debInputStream2, (OutputStream)fos);
                        break;
                    }
                }
            }
            catch (Throwable debInputStream2) {
                fileArray = debInputStream2;
                throw debInputStream2;
            }
        }
        LOGGER.debug("extracting kudu data contents");
        TarGZipUnArchiver unarchiver = new TarGZipUnArchiver();
        unarchiver.enableLogging((org.codehaus.plexus.logging.Logger)new ConsoleLogger(2, "Kudu Local Unarchive"));
        unarchiver.setSourceFile(debDataTarGz);
        unarchiver.setDestDirectory(this.kuduLocalDir);
        unarchiver.extract();
        for (File f : new File[]{debPackageFile, debDataTarGz}) {
            if (f.delete()) continue;
            LOGGER.warn("cannot delete {}", (Object)f.getAbsolutePath());
        }
        LOGGER.debug("moving kudu master and tablet binaries to {}", (Object)this.kuduLocalDir);
        Path kuduBin = Paths.get(this.kuduLocalDir.getAbsolutePath(), "usr", "lib", "kudu", "sbin-release");
        File kuduMasterBinary = kuduBin.resolve(KUDU_MASTER).toFile();
        File kuduTabletBinary = kuduBin.resolve(KUDU_TABLET).toFile();
        kuduMasterBinary.setExecutable(true);
        kuduTabletBinary.setExecutable(true);
        FileUtils.moveFileToDirectory((File)kuduMasterBinary, (File)this.kuduLocalDir, (boolean)false);
        FileUtils.moveFileToDirectory((File)kuduTabletBinary, (File)this.kuduLocalDir, (boolean)false);
        if (this.isInstalled()) {
            LOGGER.info("Kudu Local installation successful");
            return true;
        }
        LOGGER.error("Kudu Local installation failed");
        return false;
    }

    private void executeAsyncAndWatch(CommandLine command) throws ExecuteException, IOException {
        LOGGER.info("Running async: {}", (Object)command.toString());
        ExecuteWatchdog watchdog = new ExecuteWatchdog(-1L);
        DefaultExecutor executor = new DefaultExecutor();
        executor.setWatchdog(watchdog);
        executor.setWorkingDirectory(this.kuduLocalDir);
        this.watchdogs.add(watchdog);
        executor.execute(command, (ExecuteResultHandler)new DefaultExecuteResultHandler());
    }

    private void startKuduLocal() throws ExecuteException, IOException, InterruptedException {
        if (!this.kuduDBDir.exists() && !this.kuduDBDir.mkdirs()) {
            LOGGER.error("unable to create directory {}", (Object)this.kuduDBDir.getAbsolutePath());
        } else if (!this.kuduDBDir.isDirectory()) {
            LOGGER.error("{} exists but is not a directory", (Object)this.kuduDBDir.getAbsolutePath());
        }
        File kuduMasterBinary = new File(this.kuduLocalDir.getAbsolutePath(), KUDU_MASTER);
        File kuduTabletBinary = new File(this.kuduLocalDir.getAbsolutePath(), KUDU_TABLET);
        CommandLine startMaster = new CommandLine(kuduMasterBinary.getAbsolutePath());
        startMaster.addArgument("--fs_data_dirs");
        startMaster.addArgument(new File(this.kuduDBDir, "master_fs_data").getAbsolutePath());
        startMaster.addArgument("--fs_metadata_dir");
        startMaster.addArgument(new File(this.kuduDBDir, "master_fs_metadata").getAbsolutePath());
        startMaster.addArgument("--fs_wal_dir");
        startMaster.addArgument(new File(this.kuduDBDir, "master_fs_wal").getAbsolutePath());
        this.executeAsyncAndWatch(startMaster);
        for (int i = 0; i < this.numTablets; ++i) {
            CommandLine startTablet = new CommandLine(kuduTabletBinary.getAbsolutePath());
            startTablet.addArgument("--fs_data_dirs");
            startTablet.addArgument(new File(this.kuduDBDir, "t" + i + "_fs_data").getAbsolutePath());
            startTablet.addArgument("--fs_metadata_dir");
            startTablet.addArgument(new File(this.kuduDBDir, "t" + i + "_fs_metadata").getAbsolutePath());
            startTablet.addArgument("--fs_wal_dir");
            startTablet.addArgument(new File(this.kuduDBDir, "t" + i + "_fs_wal").getAbsolutePath());
            this.executeAsyncAndWatch(startTablet);
        }
        Thread.sleep(1500L);
    }

    public static void main(String[] args) {
        KuduLocal kudu = new KuduLocal(null, 1);
        kudu.start();
    }
}

