/*
 * Decompiled with CFR 0.152.
 */
package org.loesak.esque.core;

import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import lombok.NonNull;
import org.elasticsearch.client.RestClient;
import org.loesak.esque.core.concurrent.ElasticsearchDocumentLock;
import org.loesak.esque.core.elasticsearch.RestClientOperations;
import org.loesak.esque.core.elasticsearch.documents.MigrationRecord;
import org.loesak.esque.core.yaml.MigrationFileLoader;
import org.loesak.esque.core.yaml.model.MigrationFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Esque
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(Esque.class);
    private final MigrationFileLoader migrationLoader = new MigrationFileLoader();
    private final RestClientOperations operations;
    private final Lock lock;
    private final String migrationKey;
    private final String migrationUser;

    public Esque(@NonNull RestClient client, @NonNull String migrationKey) {
        this(client, migrationKey, null);
        if (client == null) {
            throw new NullPointerException("client is marked non-null but is null");
        }
        if (migrationKey == null) {
            throw new NullPointerException("migrationKey is marked non-null but is null");
        }
    }

    public Esque(@NonNull RestClient client, @NonNull String migrationKey, String migrationUser) {
        if (client == null) {
            throw new NullPointerException("client is marked non-null but is null");
        }
        if (migrationKey == null) {
            throw new NullPointerException("migrationKey is marked non-null but is null");
        }
        this.operations = new RestClientOperations(client, migrationKey);
        this.lock = new ElasticsearchDocumentLock(this.operations);
        this.migrationKey = migrationKey;
        this.migrationUser = migrationUser;
    }

    @Override
    public void close() throws IOException {
        try {
            this.lock.unlock();
        }
        catch (IllegalMonitorStateException illegalMonitorStateException) {
        }
        catch (Exception e) {
            log.warn("failed to release a execution lock. you may need to manually delete the lock document yourself", (Throwable)e);
        }
        try {
            this.operations.close();
        }
        catch (Exception e) {
            log.warn("failed to close rest client. this is likely not an issue", (Throwable)e);
        }
    }

    public void execute() {
        log.info("Starting esque execution");
        try {
            this.initialize();
            List<MigrationFile> files = this.migrationLoader.load();
            List<MigrationRecord> history = this.operations.getMigrationRecords();
            this.verifyStateIntegrity(files, history);
            this.runMigrations(files);
            log.info("Completed esque execution");
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to run esque execution", e);
        }
    }

    private void initialize() throws Exception {
        log.info("Initializing esque as needed");
        if (!this.operations.checkMigrationIndexExists().booleanValue()) {
            this.operations.createMigrationIndex();
        }
    }

    private void verifyStateIntegrity(List<MigrationFile> files, List<MigrationRecord> history) {
        try {
            log.info("Verifying integrity of migration state as compared to found migration files");
            if (history.size() > files.size()) {
                throw new IllegalStateException("the migration records are showing more migrations than the local system defines. did you refactor your files or use an incorrect migraiton key?");
            }
            if (history.size() > 0 && history.size() != history.get(history.size() - 1).getOrder() + 1) {
                throw new IllegalStateException("the migration records seem to be corrupt as some records appear to be missing.");
            }
            history.forEach(record -> {
                MigrationFile companion = files.stream().filter(file -> file.getMetadata().getFilename().equals(record.getFilename())).findFirst().orElseThrow(() -> new IllegalStateException(String.format("could not find migration file matching migration history record by filename [%s]", record.getFilename())));
                if (!(record.getOrder().intValue() == files.indexOf(companion) && record.getVersion().equals(companion.getMetadata().getVersion()) && record.getDescription().equals(companion.getMetadata().getDescription()) && record.getChecksum().equals(companion.getMetadata().getChecksum()) && record.getMigrationKey().equals(this.migrationKey))) {
                    throw new IllegalStateException(String.format("could not verify integrity of migration history record for filename [%s]. did you refactor your migration scripts after a previous execution?", record.getFilename()));
                }
            });
            log.info("Integrity checks passed");
        }
        catch (Exception e) {
            throw new IllegalStateException("state integrity checks failed", e);
        }
    }

    private void runMigrations(List<MigrationFile> files) {
        try {
            files.forEach(file -> {
                block8: {
                    try {
                        log.info("Attempting to acquire lock for execution");
                        if (this.lock.tryLock(5L, TimeUnit.MINUTES)) {
                            log.info("Lock acquired. Executing queries defined in migration file [{}]", (Object)file.getMetadata().getFilename());
                            if (this.operations.getMigrationRecordForMigrationFile((MigrationFile)file, this.migrationKey) != null) {
                                log.info("Migration for migration file [{}] and migration key [{}] appears to already have been executed. Skipping", (Object)file.getMetadata().getFilename(), (Object)this.migrationKey);
                            } else {
                                Instant start = Instant.now();
                                this.runMigrationForFile((MigrationFile)file);
                                Instant end = Instant.now();
                                Long duration = Duration.between(start, end).toMillis();
                                log.info("Execution complete for migration file [{}]. Took [{}] milliseconds", (Object)file.getMetadata().getFilename(), (Object)duration);
                                MigrationRecord record = new MigrationRecord(this.migrationKey, files.indexOf(file), file.getMetadata().getFilename(), file.getMetadata().getVersion(), file.getMetadata().getDescription(), file.getMetadata().getChecksum(), this.migrationUser, end, duration);
                                this.operations.createMigrationRecord(record);
                            }
                            break block8;
                        }
                        log.error("Failed to acquire lock in the allotted time period. Did a lock not get cleared as part of a previous execution?");
                        throw new IllegalStateException("failed to acquire lock");
                    }
                    catch (Exception e) {
                        throw new RuntimeException(String.format("Failed to execute queries in migration file [%s]", file.getMetadata().getFilename()), e);
                    }
                    finally {
                        log.info("Releasing execution lock");
                        this.lock.unlock();
                    }
                }
            });
        }
        catch (Exception e) {
            throw new IllegalStateException("failed to run migrations", e);
        }
    }

    private void runMigrationForFile(MigrationFile file) {
        log.info("Executing queries defined in migration file [{}]", (Object)file.getMetadata().getFilename());
        List<MigrationFile.MigrationFileRequestDefinition> requests = file.getContents().getRequests();
        for (MigrationFile.MigrationFileRequestDefinition definition : requests) {
            Integer position = requests.indexOf(definition);
            try {
                log.info("Executing query in position [{}] defined in migration file [{}]", (Object)position, (Object)file.getMetadata().getFilename());
                this.operations.executeMigrationDefinition(definition);
                log.info("Query in position [{}] defined in migration file [{}] executed successfully", (Object)position, (Object)file.getMetadata().getFilename());
            }
            catch (Exception e) {
                throw new IllegalStateException(String.format("Failed to execute query in position [%d] defined in migration file [%s]", position, file.getMetadata().getFilename()), e);
            }
        }
        log.info("Execution complete for queries defined in migration file [{}]", (Object)file.getMetadata().getFilename());
    }
}

