/*******************************************************************************
* Copyright (c) 2007 LuaJ. All rights reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
******************************************************************************/
package org.luaj.lib.j2se;

import java.lang.reflect.Array;
import java.util.HashMap;
import java.util.Map;

import org.luaj.vm.LBoolean;
import org.luaj.vm.LDouble;
import org.luaj.vm.LInteger;
import org.luaj.vm.LNumber;
import org.luaj.vm.LString;
import org.luaj.vm.LTable;
import org.luaj.vm.LUserData;
import org.luaj.vm.LValue;
import org.luaj.vm.LuaErrorException;


public class CoerceLuaToJava {

	public static interface Coercion { 
		public Object coerce( LValue value );
		public int score( LValue value );
	};
	
	private static Map COERCIONS = new HashMap();
	private static Coercion OBJECT_COERCION;
	
	static {
		Coercion boolCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return value.toJavaBoolean()? Boolean.TRUE: Boolean.FALSE;
			} 
			public int score(LValue value) {
				if ( value instanceof LBoolean || value.isNil() )
					return 0;
				if ( value instanceof LNumber )
					return 1;
				return 4;
			}
		};
		Coercion byteCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Byte( value.toJavaByte() );
			} 
			public int score(LValue value) {
				if ( value instanceof LInteger )
					return 1;
				if ( value instanceof LNumber )
					return 2;
				return 4;
			}
		};
		Coercion charCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Character( value.toJavaChar() );
			} 
			public int score(LValue value) {
				if ( value instanceof LInteger )
					return 1;
				if ( value instanceof LNumber )
					return 2;
				return 4;
			}
		};
		Coercion shortCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Short( value.toJavaShort() );
			} 
			public int score(LValue value) {
				if ( value instanceof LInteger )
					return 1;
				if ( value instanceof LNumber )
					return 2;
				return 4;
			}
		};
		Coercion intCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Integer( value.toJavaInt() );
			}
			public int score(LValue value) {
				if ( value instanceof LInteger )
					return 0;
				if ( value instanceof LNumber )
					return 1;
				if ( value instanceof LBoolean || value.isNil() )
					return 2;
				return 4;
			}
		};
		Coercion longCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Long( value.toJavaLong() );
			} 
			public int score(LValue value) {
				if ( value instanceof LInteger )
					return 1;
				if ( value instanceof LNumber )
					return 2;
				return 4;
			}
		};
		Coercion floatCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Float( value.toJavaFloat() );
			}
			public int score( LValue value ) {
				if ( value instanceof LNumber )
					return 1;
				if ( value instanceof LBoolean )
					return 2;
				return 4;
			}
		};
		Coercion doubleCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return new Double( value.toJavaDouble() );
			}
			public int score(LValue value) {
				if ( value instanceof LDouble )
					return 0;
				if ( value instanceof LNumber )
					return 1;
				if ( value instanceof LBoolean )
					return 2;
				return 4;
			}
		};
		Coercion stringCoercion = new Coercion() {
			public Object coerce(LValue value) {
				return value.toJavaString();
			} 
			public int score(LValue value) {
				if ( value instanceof LUserData )
					return 0;
				return 1;
			}
		};
		Coercion objectCoercion = new Coercion() {
			public Object coerce(LValue value) {
				if ( value instanceof LUserData )
					return ((LUserData)value).m_instance;
				if ( value instanceof LString )
					return value.toJavaString();
				if ( value instanceof LInteger )
					return new Integer(value.toJavaInt());
				if ( value instanceof LDouble )
					return new Double(value.toJavaDouble());
				if ( value instanceof LBoolean )
					return Boolean.valueOf(value.toJavaBoolean());
				if ( value.isNil() )
					return null;
				return value;
			} 
			public int score(LValue value) {
				if ( value instanceof LString )
					return 0;
				return 0x10;
			}
		};
		COERCIONS.put( Boolean.TYPE, boolCoercion );
		COERCIONS.put( Boolean.class, boolCoercion );
		COERCIONS.put( Byte.TYPE, byteCoercion );
		COERCIONS.put( Byte.class, byteCoercion );
		COERCIONS.put( Character.TYPE, charCoercion );
		COERCIONS.put( Character.class, charCoercion );
		COERCIONS.put( Short.TYPE, shortCoercion );
		COERCIONS.put( Short.class, shortCoercion );
		COERCIONS.put( Integer.TYPE, intCoercion );
		COERCIONS.put( Integer.class, intCoercion );
		COERCIONS.put( Long.TYPE, longCoercion );
		COERCIONS.put( Long.class, longCoercion );
		COERCIONS.put( Float.TYPE, floatCoercion );
		COERCIONS.put( Float.class, floatCoercion );
		COERCIONS.put( Double.TYPE, doubleCoercion );
		COERCIONS.put( Double.class, doubleCoercion );
		COERCIONS.put( String.class, stringCoercion );
		COERCIONS.put( Object.class, objectCoercion );
	}


	/** Score a single parameter, including array handling */
	private static int scoreParam(LValue a, Class c) {
		if ( a instanceof LUserData ) {
			Object o = ((LUserData) a).m_instance;
			if ( c.isAssignableFrom(o.getClass()) )
				return 0;
		}
		Coercion co = (Coercion) COERCIONS.get( c );
		if ( co != null ) {
			return co.score( a );
		}
		if ( c.isArray() ) {
			Class typ = c.getComponentType();
			if ( a instanceof LTable ) {
				return scoreParam( ((LTable)a).get(1), typ );
			} else {
				return 0x10 + (scoreParam(a, typ) << 8);
			}
		}
		return 0x1000;
	}

	/** Do a conversion */
	public static Object coerceArg(LValue a, Class c) {
		if ( a instanceof LUserData ) {
			Object o = ((LUserData) a).m_instance;
			if ( c.isAssignableFrom(o.getClass()) )
				return o;
		}
		Coercion co = (Coercion) COERCIONS.get( c );
		if ( co != null ) {
			return co.coerce( a );
		}
		if ( c.isArray() ) {
			boolean istable = (a instanceof LTable);
			int n = istable? a.luaLength(): 1;
			Class typ = c.getComponentType();
			Object o = Array.newInstance(typ, n);
			for ( int i=0; i<n; i++ ) {
				LValue ele = (istable? ((LTable)a).get(i+1): a);
				if ( ele != null )
					Array.set(o, i, coerceArg(ele, typ));				
			}
			return o;
		}
		if ( a.isNil() )
			return null;
		throw new LuaErrorException("no coercion found for "+a.getClass()+" to "+c);
	}

	static Object[] coerceArgs(LValue[] suppliedArgs, Class[] parameterTypes) {
		int nargs = suppliedArgs.length;
		int n = parameterTypes.length;
		Object[] args = new Object[n];
		for ( int i=0; i<n && i<nargs; i++ )
			args[i] = coerceArg( suppliedArgs[i], parameterTypes[i] );
		return args;
	}

	/*
	 * Score parameter types for match with supplied parameter list
	 * 
	 * 1) exact number of args
	 * 2) java has more args
	 * 3) java has less args
	 * 4) types coerce well
	 */
	public static int scoreParamTypes(LValue[] suppliedArgs, Class[] paramTypes) {
		int nargs = suppliedArgs.length;
		int njava = paramTypes.length;
		int score = (njava == nargs? 0: njava > nargs? 0x4000: 0x8000);
		for ( int i=0; i<nargs && i<njava; i++ ) {
			LValue a = suppliedArgs[i];
			Class c = paramTypes[i];
			int s = scoreParam( a, c );
			score += s;
		}
		return score;
	}

}
