package org.lwapp.commons.cli;

import java.io.Closeable;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CliServerFactory {
	private static final Logger LOG = LoggerFactory.getLogger(CliServerFactory.class);

	static final ExecutorService iExecutor = Executors.newCachedThreadPool();

	public static void startCli(final int port, final CliClientHandler cliHandler) throws IOException {
		final CliServer server = new CliServer(new ServerSocket(port), cliHandler);
		iExecutor.submit(server);
	}

	private static class CliServer implements Runnable, Closeable {

		private final ServerSocket serverSocket;
		private final CliClientHandler clientHandler;

		public CliServer(final ServerSocket ss, final CliClientHandler cch) {
			serverSocket = ss;
			clientHandler = cch;
		}

		@Override
		public void run() {
			LOG.info("Cli Server is starting.");
			while (!serverSocket.isClosed()) {
				LOG.info("Ready to accept cli clients to connect. Only clients from '127.0.0.1' are allowed.");
				try {
					final Socket socket = serverSocket.accept();
					final String inetHostAddress = socket.getInetAddress().getHostAddress();
					if (inetHostAddress.equals("127.0.0.1")) {
						iExecutor.submit(new CliSocketHandler(clientHandler, socket));
					} else {
						LOG.warn("Only clients from '127.0.0.1' are allowed.", inetHostAddress);
						socket.close();
					}
				} catch (final Exception e) {
					LOG.error(e.getMessage(), e);
				}
			}
		}

		@Override
		public void close() {
			try {
				if (!serverSocket.isClosed()) {
					serverSocket.close();
				}
			} catch (final IOException e) {
				LOG.warn("Failed to close socket. Try TERM KILL", e);
			}
		}

	}
}