package org.lwapp.commons;

import java.io.Serializable;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.lwapp.commons.utils.EmailValidator;

public final class Identity implements Serializable {
    private static final long serialVersionUID = 1L;

    public static final String IDENTITY_REG_EXP = "^([\\x00-\\x7F]+:[a-zA-Z]+$)";

    private final String id;
    private final IdentityType identityType;

    private Identity(final String id, final IdentityType identityType) {
        if (StringUtils.isBlank(id) || (identityType == null)) {
            throw new IllegalArgumentException("Invalid Identity or IdentityType missing.");
        }
        this.id = id;
        this.identityType = identityType;
        if (!this.toString().matches(IDENTITY_REG_EXP)) {
            throw new RuntimeException("Invalid Identity");
        }
    }

    public static Identity createIdentity(final String identity, final IdentityType type) {
        if (StringUtils.isBlank(identity)) {
            throw new IllegalArgumentException("Identity is missing");
        }
        if (type == null) {
            throw new IllegalArgumentException("IdentityType is missing");
        }

        validate(identity, type);

        return new Identity(identity, type);
    }

    private static void validate(final String identity, final IdentityType type) {
        switch (type) {
        case EMAIL: {
            if (!EmailValidator.isValid(identity)) {
                throw new IllegalArgumentException("Invalid email identity.");
            }
            break;
        }
        case MSISDN: {
            for (int i = 0; i < identity.length(); i++) {
                if (!Character.isDigit(identity.charAt(i))) {
                    throw new IllegalArgumentException("Invalid msisdn identity.");
                }
            }
            break;
        }
        default:
            break;
        }
    }

    public static Identity createMsisdnIdentity(final String msisdn) {
        return createIdentity(msisdn, IdentityType.MSISDN);
    }

    public static Identity createUidIdentity(final String id) {
        return createIdentity(id, IdentityType.UID);
    }

    public static Identity createEmailIdentity(final String email) {
        return createIdentity(email, IdentityType.EMAIL);
    }

    public static Identity createExternalIdentity(final String extId) {
        return createIdentity(extId, IdentityType.EXT);
    }

    public static Identity createUsernameIdentity(final String username) {
        return createIdentity(username, IdentityType.USERNAME);
    }

    public static Identity createMachineIdentity(final String machineName) {
        return createIdentity(machineName, IdentityType.MACHINE);
    }

    public static Identity createIdentity(final String identityString) {
        Validate.notBlank(identityString, "Identity is missing");
        final String[] identityArray = identityString.split(":");
        Validate.isTrue(identityArray.length == 2,
                "Invalid identity:" + identityString + ". Expected format:- <email>:EMAIL, <msisdn>:MSISDN etc. Valid types: " + IdentityType.values());
        final String identity = identityArray[0];
        final String identityType = identityArray[1];

        Validate.isTrue(IdentityType.isValidIdentityType(identityType), "Invalid identity type :" + identityType + ". Valid types are:" + IdentityType.values());

        return createIdentity(identity, IdentityType.valueOf(identityType));
    }

    public String getId() {
        return this.id;
    }

    public IdentityType getIdentityType() {
        return this.identityType;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = (prime * result) + (this.id == null ? 0 : this.id.hashCode());
        result = (prime * result) + (this.identityType == null ? 0 : this.identityType.hashCode());
        return result;
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        final Identity other = (Identity) obj;
        if (this.id == null) {
            if (other.id != null) {
                return false;
            }
        } else if (!this.id.equals(other.id)) {
            return false;
        }
        if (this.identityType != other.identityType) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return this.id + ":" + this.identityType.name();
    }

}
