package org.lwapp.jms.common.incoming;

import java.io.Serializable;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.PostConstruct;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.Session;
import javax.jms.TextMessage;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.lwapp.jms.common.JmsDestination;
import org.lwapp.jms.common.JmsMessage;
import org.lwapp.jms.utils.Generics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.sun.messaging.ConnectionConfiguration;

public abstract class AbstractJmsQueuePoller<T extends Serializable> {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractJmsQueuePoller.class);

    private final AtomicBoolean isAlive = new AtomicBoolean(false);

    private final AtomicLong consecutiveErrors = new AtomicLong(0);

    protected abstract JmsDestination getInJmsDestination();

    protected abstract JmsDestination getErrorQueueJmsDestination();

    protected abstract void afterReadingMessage(T jmsObject) throws Exception;

    protected void onExceptionMessage(final T jmsObject) {
    }

    @PostConstruct
    public void init() {
        try {
            final com.sun.messaging.jms.Connection connection;
            final Session session;
            final MessageConsumer consumer;
            final com.sun.messaging.ConnectionFactory cf = new com.sun.messaging.ConnectionFactory();
            cf.setProperty(ConnectionConfiguration.imqAddressList, getInJmsDestination().getUrl());

            connection = (com.sun.messaging.jms.Connection) cf.createConnection();

            // Set the Exception listner
            connection.setExceptionListener(ignore -> {
                LOG.error("Exception occured in JMS Broker. Message Details:{}", ignore.toString());
                isAlive.set(false);
            });

            // Set the Event listner
            connection.setEventListener(connEvent -> {
                final String eventCode = connEvent.getEventCode();
                final String eventMessage = connEvent.getEventMessage();

                if (StringUtils.isNotBlank(eventCode)) {
                    if (eventCode.startsWith("E2") || eventCode.equals("E401")) {
                        LOG.error("JMS broker is DISCONNECTED.");
                        isAlive.set(false);

                    } else if (eventCode.equals("E301")) {
                        LOG.info("CONNECTED successfully to JMS broker.");
                        isAlive.set(true);
                    }
                }

                LOG.info("Received JMS notification event. EventCode={}, EventMessage={}. JMS Connection={}", eventCode, eventMessage, isAlive() ? "Connected" : "Disconnected");

            });

            connection.start();
            session = connection.createSession(true, Session.SESSION_TRANSACTED);
            final String queueName = getInJmsDestination().getQueueName();
            final Queue queue = session.createQueue(queueName);
            consumer = session.createConsumer(queue);
            consumer.setMessageListener(message -> {
                LOG.info("Processing JMS message from queuename:{}", queueName);
                onMessageArrive(session, message);
            });
            isAlive.set(true);
        } catch (final Exception e) {
            throw new RuntimeException(e);
        }
    }

    public boolean isAlive() {
        return isAlive.get();
    }

    public void shutdown() {
        isAlive.set(false);
    }

    @SuppressWarnings("unchecked")
    private T unMarshalMessage(final Message message) throws Exception {
        final TextMessage textMessage = (TextMessage) message;
        final ObjectMapper objectMapper = new ObjectMapper();
        final JmsMessage jmsMessage = objectMapper.readValue(textMessage.getText(), JmsMessage.class);
        // Validate the fingerprint of the jms message
        if (!DigestUtils.sha512Hex(jmsMessage.getMessage()).equals(jmsMessage.getFingerPrint())) {
            throw new JMSException("Invalid fingerprint of the jms message. Please make sure the jms message is not altered.");
        }

        final Class<?> typeParameter = Generics.getTypeParameter(getClass());
        if (String.class.equals(typeParameter)) {
            return (T) jmsMessage.getMessage();
        }
        return (T) objectMapper.readValue(jmsMessage.getMessage(), typeParameter);
    }

    private final void onMessageArrive(final Session session, final Message message) {
        T jmsObject = null;
        try {
            jmsObject = unMarshalMessage(message);
            afterReadingMessage(jmsObject);
            consecutiveErrors.set(0);
        } catch (final Exception e) {
            try {
                onExceptionMessage(jmsObject);
            } catch (final Exception ignore) {
            }
            sendToErrorQueue(session, message, e);
            LOG.warn("Exception while processing jms message.", e);
            LOG.warn("Consecutive error count = {}.", consecutiveErrors.incrementAndGet());
        } finally {
            try {
                session.commit();
            } catch (final JMSException ignore) {
                LOG.warn("Exception occured while commiting session.", ignore);
                rollbackLogException(session);
            }
        }
    }

    private void sendToErrorQueue(final Session session, final Message message, final Exception e) {
        LOG.info("Attempting to send mail before placing message on error queue.");
        final StringBuilder sb = new StringBuilder();
        try {
            sb.append("Could not process a jms message from queue:" + getInJmsDestination().toString());
            sb.append("Exception is: ");
            sb.append(e.toString());
            sb.append("\nThe message will be moved to the error queue.\n");
            sb.append("\nJMSMessageID:" + message.getJMSMessageID());
            sb.append("\nJMSDestination:" + message.getJMSDestination());
            sb.append("\n\n");
            sb.append("Stack trace is:\n");
            sb.append(ExceptionUtils.getStackTrace(e));
            final String errorMessage = sb.toString();
            LOG.error("Message placed on error queue.\n{}", errorMessage);
        } catch (final Exception e2) {
            LOG.error("Exception occured in notifyMaxConsecutiveErrorsReached: ", e2);
        } finally {
            sendToErrorQueue(session, message);
        }

    }

    private void sendToErrorQueue(final Session session, final Message message) {
        final String errorQueue = getErrorQueueJmsDestination().getQueueName();
        Queue queue = null;
        MessageProducer producer = null;
        try {
            queue = session.createQueue(errorQueue);
            producer = session.createProducer(queue);
            LOG.info("Sending JMS message to error queue {}", errorQueue);
            producer.send(message);
            LOG.info("JMS message id: '{}' successfully sent to error queue {}.", message.getJMSMessageID(), errorQueue);
        } catch (final JMSException e) {
            LOG.warn("Exception while sending message to error queue.", e);
        } finally {
            if (producer != null) {
                try {
                    producer.close();
                    LOG.info("JMS producer closed.");
                } catch (final JMSException e) {
                    LOG.warn("Exception while closing producer.", e);
                }
            }
        }
    }

    private static void rollbackLogException(final Session session) {
        try {
            LOG.info("Rolling back session.");
            session.rollback();
        } catch (final JMSException e) {
            LOG.warn("Exception while rolling back session.", e);
        }
    }

}
