package org.lwapp.jms.common.outgoing;

import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.Session;
import javax.jms.TextMessage;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.Validate;
import org.lwapp.jms.common.JmsMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;

class JmsMessageQueueSender implements JmsCallback<Class<Void>> {
	private static final Logger LOG = LoggerFactory.getLogger(JmsMessageQueueSender.class);

	private final String queueName;
	private final String[] messages;

	JmsMessageQueueSender(final String queueName, final String... messages) {
		Validate.notBlank(queueName, "Queue name is mandatory.");
		Validate.notEmpty(messages, "There must be atleast one message to send to the queue.");
		this.queueName = queueName;
		this.messages = messages;
	}

	@Override
	public void doInJmsSession(final Session session) throws Exception {
		final Queue queue = session.createQueue(queueName);
		final MessageProducer producer = session.createProducer(queue);
		try {
			LOG.debug("Sending {} JMS message(s) to '{}' queue.", messages.length, queueName);
			for (final String message : messages) {
				final JmsMessage jmsMessage = new JmsMessage.Builder()//
						.message(message)//
						.fingerPrint(DigestUtils.sha512Hex(message))//
						.build();
				LOG.debug("Send message.");
				final ObjectMapper objectMapper = new ObjectMapper();
				final String jmsJsonMessage = objectMapper.writeValueAsString(jmsMessage);
				final TextMessage textMessage = session.createTextMessage(jmsJsonMessage);
				textMessage.setBooleanProperty("JMS_SUN_COMPRESS", true);
				producer.send(textMessage);
			}
			LOG.debug("JMS messages sent.");
		} finally {
			producer.close();
			LOG.debug("JMS producer closed.");
		}
		session.commit();
	}
}
