package org.lwapp.jms.common.utils;

import java.io.Serializable;

import javax.jms.Connection;
import javax.jms.Session;

import org.lwapp.jms.common.JmsDestination;
import org.lwapp.jms.common.outgoing.JmsCallback;
import org.lwapp.jms.common.outgoing.JmsCallbacks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sun.messaging.ConnectionConfiguration;
import com.sun.messaging.ConnectionFactory;

public class OpenMqTools {
    private static final Logger LOG = LoggerFactory.getLogger(OpenMqTools.class);

    public static void sendJmsMessage(final JmsDestination jmsDestination, final String message) throws Exception {
        sendJmsMessages(jmsDestination, message);
    }

    public static void sendJmsMessages(final JmsDestination jmsDestination, final String... messages) throws Exception {
        runInTransactedSession(jmsDestination.getUrl(), JmsCallbacks.createJmsMessageQueueSender(jmsDestination.getQueueName(), messages));
    }

    public static void publishJmsMessage(final JmsDestination jmsDestination, final String message) throws Exception {
        sendJmsMessages(jmsDestination, message);
    }

    public static void publishJmsMessages(final JmsDestination jmsDestination, final String... messages) throws Exception {
        runInTransactedSession(jmsDestination.getUrl(), JmsCallbacks.createJmsMessageTopicSender(jmsDestination.getTopicName(), messages));
    }

    private static <T extends Serializable> void runInTransactedSession(final String jmsUrl, final JmsCallback<T> jmsCallback) {
        Connection c = null;
        try {
            LOG.debug("Creating connection to {}.", jmsUrl);
            final ConnectionFactory f = new ConnectionFactory();
            f.setProperty(ConnectionConfiguration.imqAddressList, jmsUrl);
            c = f.createConnection();
            LOG.debug("Starting connection.");
            c.start();
            try {
                final Session session = c.createSession(true, Session.SESSION_TRANSACTED);
                try {
                    jmsCallback.doInJmsSession(session);
                } catch (final Exception e) {
                    try {
                        LOG.info("Exception occured while processing. Rolling back session.", e);
                        session.rollback();
                    } catch (final Exception e1) {
                        LOG.warn("Exception while rolling back session: {}", e.getMessage(), e);
                    }
                    throw e;
                } finally {
                    session.close();
                    LOG.debug("Session closed.");
                }
            } finally {
                LOG.debug("Stopping connection.");
                c.stop();
            }
        } catch (final Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (c != null) {
                try {
                    c.close();
                } catch (final Exception e) {
                }
            }
            LOG.debug("Connection closed to {}.", jmsUrl);
        }
    }

}
