package org.lwapp.notification.email;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicLong;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.inject.Inject;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.SendFailedException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.lwapp.notification.config.CommonNotificationsApplicationConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class EmailSender {
	private static final Logger LOG = LoggerFactory.getLogger(EmailSender.class);
	private final AtomicLong counter = new AtomicLong(1);
	public static final String LF = System.getProperty("line.separator", "\n");

	@Inject
	private CommonNotificationsApplicationConfig configurationClient;

	/**
	 * Wrapper method that takes an Email object as argument and immediately
	 * sends the message to the SMTP server.
	 */
	public void sendEmailNow(final Email aEmail) throws Exception {
		sendEmailUsingJavaMail(aEmail.getToAddresses(), aEmail.getSubject(), aEmail.getMessageBody(), aEmail.getAttachments());
	}

	private void sendEmailUsingJavaMail(final String[] toAddresses, final String subject, final String messageBody, final EmailAttachment[] aAttachments) throws Exception {
		Validate.notEmpty(toAddresses, "To email address is mandatory.");
		Validate.notEmpty(subject, "Subject is mandatory.");
		Validate.notEmpty(messageBody, "Email messageBody is mandatory.");

		final Properties mailServerProperties = new Properties();
		mailServerProperties.put("mail.smtp.port", "587");
		mailServerProperties.put("mail.smtp.auth", "true");
		mailServerProperties.put("mail.smtp.starttls.enable", "true");

		final Session getMailSession = Session.getInstance(mailServerProperties, new javax.mail.Authenticator() {
			@Override
			protected PasswordAuthentication getPasswordAuthentication() {
				return new PasswordAuthentication(configurationClient.getSystemEmailAddress(), configurationClient.getsystemEmailPassword());
			}
		});

		final String aFromAddress = "no-reply@boridand.com";
		List<File> tempAttachmentFiles = null;

		try {
			LOG.debug("Building email message...");
			final MimeMessage mimeMessage = new MimeMessage(getMailSession);
			mimeMessage.setFrom(new InternetAddress(aFromAddress));
			mimeMessage.setReplyTo(new InternetAddress[] { new InternetAddress(aFromAddress) });
			final List<InternetAddress> address = convertEmailAddressesToInterNetAddresses(toAddresses);
			mimeMessage.setRecipients(Message.RecipientType.TO, address.toArray(new InternetAddress[] {}));
			mimeMessage.setSubject(subject);
			mimeMessage.setSentDate(new Date());
			mimeMessage.setText(messageBody);
			// attachment currently unsupported
			if ((aAttachments != null) && (aAttachments.length > 0)) {
				tempAttachmentFiles = addAttachments(aAttachments, mimeMessage, messageBody);
			}

			LOG.debug("Sending message to " + StringUtils.join(toAddresses, " "));
			final Transport transport = getMailSession.getTransport("smtp");
			try {
				transport.connect("smtp.gmail.com", "", "");
				transport.sendMessage(mimeMessage, mimeMessage.getAllRecipients());
			} finally {
				if (transport != null) {
					transport.close();
				}
			}

			LOG.info("Message sent successfully!");
		} catch (final MessagingException mex) {
			LOG.error("Could not send messages to all recepients.", mex);
			// There may be several exceptions so we need a temporary exception
			// variable to use in this loop
			logSendProblems(mex, LOG);
			throw mex;
		} finally {
			try {
				if (tempAttachmentFiles != null) {
					for (final File file : tempAttachmentFiles) {
						file.delete();
					}
				}
			} catch (final Exception ex) {
				LOG.debug("Ignore. Could not delete the file. Delete the temp files manually", ex);
			}
		}
	}

	private void logSendProblems(final MessagingException mex, final Logger aLogger) {
		Exception ex = mex;
		do {
			if (ex instanceof SendFailedException) {
				final SendFailedException sfex = (SendFailedException) ex;
				final Address[] invalid = sfex.getInvalidAddresses();
				if (invalid != null) {
					aLogger.error("    ** Invalid Addresses");
					for (final Address element : invalid) {
						aLogger.error("         " + element);
					}
				}
				final Address[] validUnsent = sfex.getValidUnsentAddresses();
				if (validUnsent != null) {
					aLogger.error("    ** ValidUnsent Addresses");
					for (final Address element : validUnsent) {
						aLogger.error("         " + element);
					}
				}
				final Address[] validSent = sfex.getValidSentAddresses();
				if (validSent != null) {
					aLogger.error("    ** ValidSent Addresses");
					for (final Address element : validSent) {
						aLogger.error("         " + element);
					}
				}
			}
			if (ex instanceof MessagingException) {
				ex = ((MessagingException) ex).getNextException();
			} else {
				ex = null;
			}
		} while (ex != null);
	}

	private List<File> addAttachments(final EmailAttachment[] emailAttachments, final Message aMsg, final String aMessageBody) throws Exception {

		final List<File> temperoryAttachmentFiles = new ArrayList<>();
		if (emailAttachments == null) {
			return temperoryAttachmentFiles;
		}

		String attachmentErrorStr = "";
		final BodyPart messageBodyPart = new MimeBodyPart();
		messageBodyPart.setText(aMessageBody);

		final Multipart multipart = new MimeMultipart();
		multipart.addBodyPart(messageBodyPart);

		// Part two is attachment
		for (final EmailAttachment emailAttachment : emailAttachments) {
			if (emailAttachment != null) {
				final String attachmentName = emailAttachment.getAttachmentName();
				final File fileToAttach = File.createTempFile(String.valueOf(counter.incrementAndGet()), "");
				temperoryAttachmentFiles.add(fileToAttach);
				FileUtils.writeByteArrayToFile(fileToAttach, emailAttachment.getByteAttay());
				if (!fileToAttach.exists()) {
					throw new RuntimeException("Could not able to create attachment file at: " + fileToAttach.getParent());
				}

				try {
					final BodyPart mimeBodyPart = new MimeBodyPart();
					final DataSource source = new FileDataSource(fileToAttach);
					mimeBodyPart.setDataHandler(new DataHandler(source));
					mimeBodyPart.setFileName(attachmentName);
					multipart.addBodyPart(mimeBodyPart);
				} catch (final MessagingException attachmentEx) {
					attachmentErrorStr += LF + attachmentName;
					LOG.warn("Failed to attach: " + attachmentName, attachmentEx);
					continue;
				}
			}
		}

		if (!StringUtils.isEmpty(attachmentErrorStr)) {
			messageBodyPart.setText(aMessageBody + LF + "Faild to attach the following files: " + attachmentErrorStr);
		}
		aMsg.setContent(multipart);
		aMsg.saveChanges();
		return temperoryAttachmentFiles;
	}

	/**
	 * Converts an array of Strings containing email addresses to an array of
	 * InternetAddress objects to be used by JavaMail.
	 */
	private static List<InternetAddress> convertEmailAddressesToInterNetAddresses(final String[] aToAddresses) throws AddressException {
		Validate.notEmpty(aToAddresses, "To email address is mandatory.");
		final List<InternetAddress> result = new ArrayList<InternetAddress>();
		for (final String toAddress : aToAddresses) {
			result.add(new InternetAddress(toAddress));
		}
		return result;
	}

}
