package org.lwapp.notification.jms.outgoing;

import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.jms.Topic;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.Validate;
import org.lwapp.notification.jms.JmsMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.fasterxml.jackson.databind.ObjectMapper;

class JmsMessageTopicSender implements JmsCallback<Class<Void>> {
    private static final Logger LOG = LoggerFactory.getLogger(JmsMessageTopicSender.class);

    private final String topicName;
    private final String[] messages;

    JmsMessageTopicSender(final String topicName, final String... messages) {
        Validate.notBlank(topicName, "Queue name is mandatory.");
        Validate.notEmpty(messages, "There must be atleast one message to send to the queue.");
        this.topicName = topicName;
        this.messages = messages;
    }

    @Override
    public Class<Void> doInJmsSession(final Session session) throws Exception {
        final Topic topic = session.createTopic(topicName);
        final MessageProducer producer = session.createProducer(topic);
        try {
            LOG.debug("Sending {} JMS message(s) to '{}' topic.", messages.length, topicName);
            for (final String message : messages) {
                final JmsMessage jmsMessage = new JmsMessage();
                jmsMessage.setMessage(message);
                jmsMessage.setFingerPrint(DigestUtils.sha512Hex(message));
                jmsMessage.setLogTraceId(MDC.get("logTraceId"));
                LOG.debug("Send message.");
                final ObjectMapper objectMapper = new ObjectMapper();
                final String jmsJsonMessage = objectMapper.writeValueAsString(jmsMessage);
                final TextMessage textMessage = session.createTextMessage(jmsJsonMessage);
                textMessage.setBooleanProperty("JMS_SUN_COMPRESS", true);
                producer.send(textMessage);
            }
            LOG.debug("JMS messages sent.");
        } finally {
            producer.close();
            LOG.debug("JMS producer closed.");
        }
        session.commit();
        return Void.TYPE;
    }
}
