package org.lwapp.notification.jms.utils;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Session;

import org.lwapp.notification.jms.JmsDestination;
import org.lwapp.notification.jms.JmsMessage;
import org.lwapp.notification.jms.outgoing.JmsCallback;
import org.lwapp.notification.jms.outgoing.JmsCallbacks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sun.messaging.ConnectionConfiguration;
import com.sun.messaging.ConnectionFactory;

public class OpenMqTools {
    private static final Logger LOG = LoggerFactory.getLogger(OpenMqTools.class);

    public static JmsMessage receiveMessage(final JmsDestination jmsDestination) throws Exception {
        return runInTransactedSession(jmsDestination.getHost(), JmsCallbacks.createMessageQueueReceiver(jmsDestination.getQueueName()));
    }

    public static void sendJmsMessage(final JmsDestination jmsDestination, final String message) throws Exception {
        sendJmsMessages(jmsDestination, message);
    }

    public static void sendJmsMessages(final JmsDestination jmsDestination, final String... messages) throws Exception {
        runInTransactedSession(jmsDestination.getHost(), JmsCallbacks.createJmsMessageQueueSender(jmsDestination.getQueueName(), messages));
    }

    public static void publishJmsMessage(final JmsDestination jmsDestination, final String message) throws Exception {
        sendJmsMessages(jmsDestination, message);
    }

    public static void publishJmsMessages(final JmsDestination jmsDestination, final String... messages) throws Exception {
        runInTransactedSession(jmsDestination.getHost(), JmsCallbacks.createJmsMessageTopicSender(jmsDestination.getTopicName(), messages));
    }

    private static <T> T runInTransactedSession(final String jmsUrl, final JmsCallback<T> jmsCallback) throws Exception {
        LOG.debug("Creating connection to {}.", jmsUrl);
        final Connection c = createConnection(jmsUrl);
        try {
            LOG.debug("Starting connection.");
            c.start();
            try {
                return createTransactedSessionAndRun(c, jmsCallback);
            } finally {
                LOG.debug("Stopping connection.");
                c.stop();
            }
        } finally {
            c.close();
            LOG.debug("Connection closed to {}.", jmsUrl);
        }
    }

    private static Connection createConnection(final String jmsUrl) throws JMSException {
        final ConnectionFactory f = new ConnectionFactory();
        f.setProperty(ConnectionConfiguration.imqAddressList, jmsUrl);
        return f.createConnection();
    }

    private static <T> T createTransactedSessionAndRun(final Connection c, final JmsCallback<T> jmsCallback) throws Exception {
        final Session session = c.createSession(true, Session.SESSION_TRANSACTED);
        try {
            return jmsCallback.doInJmsSession(session);
        } catch (final Exception e) {
            rollbackLogException(session);
            throw e;
        } finally {
            session.close();
            LOG.debug("Session closed.");
        }
    }

    private static void rollbackLogException(final Session session) {
        try {
            LOG.info("Rolling back session.");
            session.rollback();
        } catch (final Exception e) {
            LOG.warn("Exception while rolling back session: {}", e.getMessage(), e);
        }
    }

}
