package org.lwapp.notification;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.lwapp.notification.config.CommonNotificationsApplicationConfig;
import org.lwapp.notification.email.Email;
import org.lwapp.notification.email.EmailAttachment;
import org.lwapp.notification.jms.utils.OpenMqTools;
import org.lwapp.notification.sms.Sms;
import org.lwapp.notification.twitter.Tweet;

import com.fasterxml.jackson.databind.ObjectMapper;

@Singleton
public final class NotificationManager {

	@Inject
	private CommonNotificationsApplicationConfig commonNotificationsApplicationConfig;

	public void sendEmail(final String subject, final String message, final String... recipients) throws Exception {
		if (!isEmailNotificationsEnabled()) {
			return;
		}

		final Email email = new Email(subject, message, recipients);
		final ObjectMapper objectMapper = new ObjectMapper();
		final String emailJson = objectMapper.writeValueAsString(email);
		OpenMqTools.sendJmsMessage(commonNotificationsApplicationConfig.getEmailIncomingJmsQueue(), emailJson);
	}

	public void sendEmail(final String subject, final String message, final List<String> recipients, final EmailAttachment... attachments) throws Exception {
		if (!isEmailNotificationsEnabled()) {
			return;
		}
		final Email email = new Email(recipients, subject, message, attachments);
		final ObjectMapper objectMapper = new ObjectMapper();
		final String emailJson = objectMapper.writeValueAsString(email);
		OpenMqTools.sendJmsMessage(commonNotificationsApplicationConfig.getEmailIncomingJmsQueue(), emailJson);
	}

	public void sendSms(final String msisdn, final String message) throws Exception {
		if (!isSmsNotificationsEnabled()) {
			return;
		}
		final Sms sms = new Sms(message, msisdn);
		final ObjectMapper objectMapper = new ObjectMapper();
		final String smsJson = objectMapper.writeValueAsString(sms);
		OpenMqTools.sendJmsMessage(commonNotificationsApplicationConfig.getSmsIncomingJmsQueue(), smsJson);
	}

	public void tweet(final String to, final String message) throws Exception {

		if (!isTwitterNotificationsEnabled()) {
			return;
		}
		final Tweet tweet = new Tweet(to, message);
		final ObjectMapper objectMapper = new ObjectMapper();
		final String tweetJson = objectMapper.writeValueAsString(tweet);
		OpenMqTools.sendJmsMessage(commonNotificationsApplicationConfig.getTwitterIncomingJmsQueue(), tweetJson);
	}

	private boolean isEmailNotificationsEnabled() {
		return commonNotificationsApplicationConfig.isNotificationsEnabled() && commonNotificationsApplicationConfig.isEmailNotificationsEnabled();
	}

	private boolean isSmsNotificationsEnabled() {
		return commonNotificationsApplicationConfig.isNotificationsEnabled() && commonNotificationsApplicationConfig.isSmsNotificationsEnabled();
	}

	private boolean isTwitterNotificationsEnabled() {
		return commonNotificationsApplicationConfig.isNotificationsEnabled() && commonNotificationsApplicationConfig.isTwitterNotificationsEnabled();
	}

}
