package org.lwapp.notification.jms.outgoing;

import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.Queue;
import javax.jms.Session;
import javax.jms.TextMessage;

import org.apache.commons.codec.digest.DigestUtils;
import org.lwapp.notification.jms.JmsMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;

class JmsMessageQueueReceiver implements JmsCallback<JmsMessage> {
    private static final Logger LOG = LoggerFactory.getLogger(JmsMessageQueueReceiver.class);

    private final String queueName;
    private final long timeoutMillis;

    JmsMessageQueueReceiver(final String queueName, final long timeoutMillis) {
        this.queueName = queueName;
        this.timeoutMillis = timeoutMillis;
    }

    @Override
    public JmsMessage doInJmsSession(final Session session) throws Exception {
        final Queue queue = session.createQueue(queueName);
        final MessageConsumer consumer = session.createConsumer(queue);
        final JmsMessage jmsMessage;
        LOG.debug("Consumer created for queue '{}'.", queueName);
        try {
            final Message message = consumer.receive(timeoutMillis);
            final TextMessage textMessage = (TextMessage) message;
            final ObjectMapper objectMapper = new ObjectMapper();
            jmsMessage = objectMapper.readValue(textMessage.getText(), JmsMessage.class);
            // Validate the fingerprint of the jms message
            if (!DigestUtils.sha512Hex(jmsMessage.getMessage()).equals(jmsMessage.getFingerPrint())) {
                throw new Exception("Invalid fingerprint of the jms message. Please make sure the jms message is not altered.");
            }

        } finally {
            consumer.close();
            LOG.debug("Consumer closed.");
        }
        session.commit();
        return jmsMessage;
    }
}
