package org.lwapp.security.ws.filter;

import java.io.IOException;

import javax.inject.Inject;
import javax.interceptor.Interceptors;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.ext.Provider;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.Validate;
import org.lwapp.commons.exception.UnauthorizedException;
import org.lwapp.hibernate.interceptor.UnitOfWork;
import org.lwapp.hibernate.interceptor.UnitOfWorkInterceptor;
import org.lwapp.security.service.AuthorizationService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Provider
@Interceptors({ UnitOfWorkInterceptor.class })
public class SecurityFilter implements ContainerRequestFilter {
	private static final Logger LOG = LoggerFactory.getLogger(SecurityFilter.class);

	public static final String AUTHORIZATION_HEADER = "Authorization";
	public static final String APPLICATION_OWNER_ID = "ApplicationOwnerId";
	public static final String APPLICATION_NAME = "ApplicationName";

	@Inject
	private AuthorizationService authenticationService;

	@Override
	@UnitOfWork
	public void filter(final ContainerRequestContext requestContext) throws IOException {

		if (Boolean.valueOf(requestContext.getHeaderString("Internal"))) {
			return;
		}

		final String applicationOwnerId = requestContext.getHeaderString(APPLICATION_OWNER_ID);
		Validate.notBlank(applicationOwnerId, "Please provide ApplicationOwner in header.");
		final String authCredentials = requestContext.getHeaderString(AUTHORIZATION_HEADER);
		Validate.notBlank(authCredentials, "Please provide basic authentication in header.");
		final String applicationName = requestContext.getHeaderString(APPLICATION_NAME);
		Validate.notBlank(applicationName, "Please provide ApplicationName in header.");

		final boolean authenticationStatus = authenticationService.authenticate(authCredentials, applicationOwnerId, applicationName);

		if (BooleanUtils.isFalse(authenticationStatus)) {
			throw new UnauthorizedException(
					"Un-authorized application client. Make sure you have used correctly 'apiKey' and 'apiSecret' shall be part of BasicAuthentication. Please check headers for 'applicationName' and 'applicationOwnerId' entries aswell.");
		}

		LOG.info("Method:{}, applicationOwnerId:{}, applicationName:{}", requestContext.getMethod(), applicationOwnerId, applicationName);
	}

}
