package org.lwapp.security.ws.rest;

import javax.inject.Inject;
import javax.interceptor.Interceptors;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.core.Response;

import org.lwapp.commons.exception.ApplicationException;
import org.lwapp.hibernate.interceptor.UnitOfWork;
import org.lwapp.hibernate.interceptor.UnitOfWorkInterceptor;
import org.lwapp.hibernate.persistence.common.UniqueNumberGenerator;
import org.lwapp.security.db.dao.AuthorizedAppsDao;
import org.lwapp.security.db.dao.AuthorizedOwnerDao;
import org.lwapp.security.db.entity.AuthorizedApps;
import org.lwapp.security.db.entity.AuthorizedOwner;
import org.lwapp.security.ws.rest.domain.ApplicationOwnerRegistrationRequest;
import org.lwapp.security.ws.rest.domain.ApplicationOwnerRegistrationResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Path("admin")
@Interceptors({ UnitOfWorkInterceptor.class })
public class ApplicationAuthorizationResource {

	private static final Logger LOG = LoggerFactory.getLogger(ApplicationAuthorizationResource.class);

	@Inject
	private AuthorizedOwnerDao authorizedOwnerDao;
	@Inject
	private AuthorizedAppsDao authorizedAppsDao;
	@Inject
	private UniqueNumberGenerator uniqueNumberGenerator;

	@POST
	@Path("application/generateapikeys")
	@UnitOfWork
	public Response generateApiKeys(final ApplicationOwnerRegistrationRequest request) {

		final String uniqueOwnerId = request.getUniqueOwnerId();

		AuthorizedOwner authOwner = authorizedOwnerDao.findByUniqueOwnerId(uniqueOwnerId);
		if (authOwner == null) {
			authOwner = new AuthorizedOwner();
			authOwner.setOwnerName(request.getOwnerName());
			authOwner.setOwnerId(uniqueOwnerId);
			authorizedOwnerDao.persist(authOwner);
		}

		AuthorizedApps authApp = authorizedAppsDao.findByApplication(request.getApplicationName(), uniqueOwnerId);

		if (authApp == null) {
			authApp = new AuthorizedApps();
			authApp.setApiKey(uniqueNumberGenerator.generateUniqueApiKey());
			authApp.setApiSecret(uniqueNumberGenerator.generateUniqueApiSecret());
			authApp.setApplicationName(request.getApplicationName());
			authApp.setAuthorizedOwner(authOwner);
			authorizedAppsDao.persist(authApp);
		}

		final ApplicationOwnerRegistrationResponse response = new ApplicationOwnerRegistrationResponse.Builder()//
				.apiKey(authApp.getApiKey())//
				.apiSecret(authApp.getApiSecret())//
				.applicationName(request.getApplicationName())//
				.uniqueOwnerId(request.getUniqueOwnerId())//
				.build();

		LOG.info("Application owner is created succussfully with authApp \n\napiKey:{} \n\napiSecret:{} \nownerId:{} \napplicationName:{}", //
				authApp.getApiKey(), authApp.getApiSecret(), request.getUniqueOwnerId(), request.getApplicationName());

		return Response.ok(response).build();
	}

	@POST
	@Path("application/regenerateapikeys/")
	@UnitOfWork
	public Response reGenerateApiKeys(final ApplicationOwnerRegistrationRequest request) {

		final String uniqueOwnerId = request.getUniqueOwnerId();

		final AuthorizedOwner authOwner = authorizedOwnerDao.findByUniqueOwnerId(uniqueOwnerId);

		if (authOwner == null) {
			throw new ApplicationException("No application owner found with the given ownerId.");
		}

		AuthorizedApps authApp = authorizedAppsDao.findByApplication(request.getApplicationName(), uniqueOwnerId);

		final String apiKey = uniqueNumberGenerator.generateUniqueApiKey();
		final String apiSecret = uniqueNumberGenerator.generateUniqueApiSecret();

		if (authApp == null) {
			authApp = new AuthorizedApps();
			authApp.setApiKey(apiKey);
			authApp.setApiSecret(apiSecret);
			authApp.setApplicationName(request.getApplicationName());
			authApp.setAuthorizedOwner(authOwner);
			authorizedAppsDao.persist(authApp);
		} else {
			authApp.setApiKey(apiKey);
			authApp.setApiSecret(apiSecret);
			authorizedAppsDao.update(authApp);
		}

		final ApplicationOwnerRegistrationResponse response = new ApplicationOwnerRegistrationResponse.Builder()//
				.apiKey(authApp.getApiKey())//
				.apiSecret(authApp.getApiSecret())//
				.applicationName(request.getApplicationName())//
				.uniqueOwnerId(request.getUniqueOwnerId())//
				.build();

		LOG.info("Application owner api keys are recreated succussfully with authApp \n\napiKey:{} \n\napiSecret:{} \nownerId:{} \napplicationName:{}", //
				authApp.getApiKey(), authApp.getApiSecret(), request.getUniqueOwnerId(), request.getApplicationName());

		return Response.ok(response).build();
	}

}
