package org.lwapp.security.interceptor;

import javax.inject.Inject;
import javax.interceptor.AroundInvoke;
import javax.interceptor.Interceptor;
import javax.interceptor.InvocationContext;
import javax.ws.rs.core.HttpHeaders;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.Validate;
import org.lwapp.commons.exception.UnauthorizedException;
import org.lwapp.security.service.AuthorizationService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Interceptor
public class ClientAuthorizationInterceptor {
	private static final Logger LOG = LoggerFactory.getLogger(ClientAuthorizationInterceptor.class);

	public static final String AUTHORIZATION_HEADER = "Authorization";
	public static final String APPLICATION_OWNER_ID = "ApplicationOwnerId";
	public static final String APPLICATION_NAME = "ApplicationName";

	@Inject
	private AuthorizationService authenticationService;

	@AroundInvoke
	public Object aroundInvoke(final InvocationContext context) throws Exception {
		LOG.info("Executing operation {}.", context.getMethod().toGenericString());
		final AuthorizeClient authClient = context.getMethod().getAnnotation(AuthorizeClient.class);
		if (authClient != null) {
			final Object[] parameters = context.getParameters();
			final HttpHeaders httpHeaders = (HttpHeaders) parameters[authClient.httpHeaderPosition()];
			Validate.notNull(httpHeaders, "HttpHeaders is missing.");

			final String applicationOwnerId = httpHeaders.getHeaderString(APPLICATION_OWNER_ID);
			Validate.notBlank(applicationOwnerId, "Please provide 'ApplicationOwnerId' in header.");
			final String authCredentials = httpHeaders.getHeaderString(AUTHORIZATION_HEADER);
			Validate.notBlank(authCredentials, "Please provide 'Basic-Authentication' in header.");
			final String applicationName = httpHeaders.getHeaderString(APPLICATION_NAME);
			Validate.notBlank(applicationName, "Please provide 'ApplicationName' in header.");

			final boolean authenticationStatus = authenticationService.authenticate(authCredentials, applicationOwnerId, applicationName);
			LOG.info("Method:{}, applicationOwnerId:{}, applicationName:{}", context.getMethod().toGenericString(), applicationOwnerId, applicationName);

			if (BooleanUtils.isFalse(authenticationStatus)) {
				throw new UnauthorizedException(
						"Un-authorized application client. Make sure you have used correctly 'apiKey' and 'apiSecret' shall be part of BasicAuthentication. Please check headers for 'applicationName' and 'applicationOwnerId' entries aswell.");
			}
		}
		return context.proceed();
	}
}
