package org.lwapp.security.manager;

import javax.inject.Inject;

import org.lwapp.hibernate.persistence.common.UniqueNumberGenerator;
import org.lwapp.security.db.dao.AuthorizedAppsDao;
import org.lwapp.security.db.dao.AuthorizedOwnerDao;
import org.lwapp.security.db.entity.AuthorizedApps;
import org.lwapp.security.db.entity.AuthorizedOwner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApplicationAuthorizationManager {

	private static final Logger LOG = LoggerFactory.getLogger(ApplicationAuthorizationManager.class);

	@Inject
	private AuthorizedOwnerDao authorizedOwnerDao;
	@Inject
	private AuthorizedAppsDao authorizedAppsDao;
	@Inject
	private UniqueNumberGenerator uniqueNumberGenerator;

	public ApplicationOwnerRegistrationResponse generateApiKeys(final ApplicationOwnerRegistrationRequest request) {

		final String uniqueOwnerId = request.getUniqueOwnerId();

		AuthorizedOwner authOwner = authorizedOwnerDao.findByUniqueOwnerId(uniqueOwnerId);

		if (authOwner == null) {
			authOwner = new AuthorizedOwner();
			authOwner.setOwnerName(request.getOwnerName());
			authOwner.setOwnerId(uniqueOwnerId);
			authorizedOwnerDao.persist(authOwner);
		}

		AuthorizedApps authApp = authorizedAppsDao.findByApplication(request.getApplicationName(), uniqueOwnerId);

		final String apiKey = uniqueNumberGenerator.generateUniqueApiKey();
		final String apiSecret = uniqueNumberGenerator.generateUniqueApiSecret();

		if (authApp == null) {
			authApp = new AuthorizedApps();
			authApp.setApiKey(apiKey);
			authApp.setApiSecret(apiSecret);
			authApp.setApplicationName(request.getApplicationName());
			authApp.setAuthorizedOwner(authOwner);
			authorizedAppsDao.persist(authApp);
		} else {
			authApp.setApiKey(apiKey);
			authApp.setApiSecret(apiSecret);
			authorizedAppsDao.update(authApp);
		}

		final ApplicationOwnerRegistrationResponse response = new ApplicationOwnerRegistrationResponse.Builder()//
				.apiKey(authApp.getApiKey())//
				.apiSecret(authApp.getApiSecret())//
				.applicationName(request.getApplicationName())//
				.uniqueOwnerId(request.getUniqueOwnerId())//
				.build();

		LOG.info("Application owner api keys are recreated succussfully with authApp \n\napiKey:{} \n\napiSecret:{} \nownerId:{} \napplicationName:{}", //
				authApp.getApiKey(), authApp.getApiSecret(), request.getUniqueOwnerId(), request.getApplicationName());

		return response;
	}

}
