package org.magictest.eclipse;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.resources.IFile;
import org.eclipse.jdt.core.IAnnotation;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IImportDeclaration;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeRoot;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.internal.corext.dom.NodeFinder;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.jdt.internal.ui.javaeditor.JavaEditor;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jdt.ui.SharedASTProvider;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.graphics.Point;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.texteditor.ITextEditor;
import org.magictest.eclipse.tools.JdtTools;
import org.magictest.ext.ch.qos.logback.classic.Logger;
import org.magictest.ext.org.magicwerk.brownies.core.StringTools;
import org.magictest.ext.org.magicwerk.brownies.core.reflect.ClassTools;
import org.magictest.ext.org.slf4j.LoggerFactory;

/**
 * Handle command for a user event.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class CommandHandler {
	/** Logger */
	private static Logger LOG = (Logger) LoggerFactory.getLogger(CommandHandler.class);

	private ExecutionEvent event;

	/**
	 * Constructor.
	 *
	 * @param event
	 */
	public CommandHandler(ExecutionEvent event) {
		this.event = event;
	}

	public void runTest() {
		SelectInfo selectInfo = getSelectInfo();
		LOG.info("CommandHandler.runTest:" + selectInfo);
		if (selectInfo != null) {
			runTest(selectInfo);
		}
	}

	public void runProfileTest() {
		SelectInfo selectInfo = getSelectInfo();
		LOG.info("CommandHandler.runProfileTest:" + selectInfo);
		if (selectInfo != null) {
			runProfileTest(selectInfo);
		}
	}

	public void saveTest() {
		SelectInfo selectInfo = getSelectInfo();
		LOG.debug("CommandHandler.saveTest:" + selectInfo);
		if (selectInfo != null) {
			saveTest(selectInfo);
		}
	}

	/**
	 * Runs tests.
	 * Called from menu or link in HTML report.
	 */
	public static void runTest(SelectInfo selectInfo) {
		MagicTestNGLauncher.runTest(selectInfo, false);
		BrowserSupport.refreshMagicTestDir(selectInfo);
	}

	/**
	 * Runs tests in profile mode.
	 * Called from menu.
	 */
	public static void runProfileTest(SelectInfo selectInfo) {
		MagicTestNGLauncher.runTest(selectInfo, true);
	}

	/**
	 * Saves tests.
	 * Called from menu or link in HTML report.
	 */
	public static void saveTest(SelectInfo selectInfo) {
		MagicTestLauncher.saveTest(selectInfo);
		if (selectInfo.packageName != null) {
			BrowserSupport.refreshOpenHtmlFiles(selectInfo);
		} else {
			BrowserSupport.showActualReport(selectInfo);
		}
		BrowserSupport.refreshMagicTestDir(selectInfo);
	}

	/**
	 * Deletes tests.
	 * Called from link in HTML report.
	 */
	public static void deleteTest(SelectInfo selectInfo) {
		MagicTestLauncher.deleteTest(selectInfo);
		if (selectInfo.packageName != null) {
			BrowserSupport.refreshOpenHtmlFiles(selectInfo);
		} else {
			BrowserSupport.showActualReport(selectInfo);
		}
		BrowserSupport.refreshMagicTestDir(selectInfo);
	}

	/**
	 * Invalidates tests.
	 * Called from link in HTML report.
	 */
	public static void invalidateTest(SelectInfo selectInfo) {
		MagicTestLauncher.invalidateTest(selectInfo);
		if (selectInfo.packageName != null) {
			BrowserSupport.refreshOpenHtmlFiles(selectInfo);
		} else {
			BrowserSupport.showActualReport(selectInfo);
		}
		BrowserSupport.refreshMagicTestDir(selectInfo);
	}

	/**
	 * Show source code of test.
	 */
	public void showTest() {
		SelectInfo selectInfo = getSelectInfo();
		if (selectInfo != null) {
			LOG.debug("CommandHandler.showTest:" + selectInfo);
			showTest(selectInfo, true);
		}
	}

	/**
	 * Show source of test.
	 * Called from link in HTML report.
	 *
	 * @param selectInfo    selection
	 * @param search		true to search both test from testee and vice versa
	 */
	public static void showTest(SelectInfo selectInfo, boolean search) {
		if (selectInfo.className != null) {
			IMember member = null;
			ISourceRange range = null;
			if (selectInfo.methodName == null) {
				member = getType(selectInfo, search);
			} else {
				member = getMethod(selectInfo, search);
			}
			if (member != null) {
				try {
					LOG.info("Show {}", member.getElementName());
					ITextEditor textEditor = (ITextEditor) EditorUtility.openInEditor(member, true);

					range = member.getNameRange();
					textEditor.selectAndReveal(range.getOffset(), range.getLength());

				} catch (Exception e) {
					LOG.info("Show {} failed", e);
				}
			}
		}
	}

	/**
	 * Find a test method for the currently selected method and return it.
	 *
	 * @param selectInfo  current selection
	 * @param search		true to search both test from testee and vice versa
	 * @return found test method or null
	 */
	static IMember getMethod(SelectInfo selectInfo, boolean search) {
		IType type = getType(selectInfo, search);
		if (type == null) {
			return null;
		}

		if (!search) {
			String testMethodName = selectInfo.methodName;
			IMethod method = JdtTools.getTestMethod(type, testMethodName);
			if (method != null) {
				return method;
			}
			return null;

		} else {
			if (isTestType(type)) {
				String testMethodName = selectInfo.methodName + "Test";
				IMethod method = JdtTools.getTestMethod(type, testMethodName);
				if (method != null) {
					return method;
				}
				testMethodName = "test" + selectInfo.methodName.substring(0, 1).toUpperCase() + selectInfo.methodName.substring(1);
				method = JdtTools.getTestMethod(type, testMethodName);
				if (method != null) {
					return method;
				}
			} else {
				String methodName = selectInfo.methodName;
				if (methodName.endsWith("Test")) {
					methodName = StringTools.removeTail(methodName, "Test");
				} else if (methodName.startsWith("test") && methodName.length() > 4 && Character.isUpperCase(methodName.charAt(4))) {
					methodName = Character.toLowerCase(methodName.charAt(4)) + methodName.substring(5);
				}
				IMethod method = JdtTools.getTestMethod(type, methodName);
				if (method != null) {
					return method;
				}
			}
			// If we look for test method and do not find the exact method,
			// we show the test class anyhow
			return type;
		}
	}

	/**
	 * Find a test class for the currently selected class and return it.
	 *
	 * @param selectInfo    current selection
	 * @param search		true to search both test from testee and vice versa
	 * @return              found test class or null
	 */
	static IType getType(SelectInfo selectInfo, boolean search) {
		IType type;
		if (!search) {
			type = getType(selectInfo.projectName, selectInfo.className);
			if (type != null) {
				return type;
			}
		} else {
			type = getType(selectInfo.projectName, selectInfo.className);
			boolean searchTest = false;
			boolean searchTestee = false;
			if (type == null) {
				searchTest = true;
				searchTestee = true;
			} else {
				if (isTestType(type)) {
					searchTestee = true;
				} else {
					searchTest = true;
				}
			}
			if (searchTestee) {
				// First look in test project
				String prjName = selectInfo.projectName;
				IType testType = getTesteeType(prjName, selectInfo.className);
				if (testType != null) {
					return testType;
				}
				// Then in testee project
				prjName = StringTools.removeTailIf(prjName, "-Test");
				prjName = StringTools.removeTailIf(prjName, "Test");
				testType = getTesteeType(prjName, selectInfo.className);
				if (testType != null) {
					return testType;
				}
			}
			if (searchTest) {
				IType testType = getTestType(selectInfo.projectName, selectInfo.className);
				if (testType != null) {
					return testType;
				}
				testType = getTestType(selectInfo.projectName + "Test", selectInfo.className);
				if (testType != null) {
					return testType;
				}
				testType = getTestType(selectInfo.projectName + "-Test", selectInfo.className);
				if (testType != null) {
					return testType;
				}
			}
		}
		return null;
	}

	static boolean isTestType(IType type) {
		try {
			IImportDeclaration[] imports = type.getCompilationUnit().getImports();
			for (IImportDeclaration imp : imports) {
				if (isTestImport(imp)) {
					return true;
				}
			}

			for (IAnnotation annot : type.getAnnotations()) {
				if (isTestAnnotation(annot)) {
					return true;
				}
			}

			IMethod[] methods = type.getMethods();
			for (IMethod method : methods) {
				for (IAnnotation annot : method.getAnnotations()) {
					if (isTestAnnotation(annot)) {
						return true;
					}
				}
			}
			return false;
		} catch (JavaModelException e) {
			throw new RuntimeException(e);
		}
	}

	static final String MAGICTEST_ANNOTATION_PKG = "org.magictest.client.";
	static final String TESTNG_ANNOTATION_PKG = "org.testng.annotations.";
	static final String JUNIT_ANNOTATION_PKG = "org.junit.";
	static final String TRACE_ANNOTATION = "Trace";
	static final String CAPTURE_ANNOTATION = "Capture";
	static final String TEST_ANNOTATION = "Test";
	static final String ASSERT_ANNOTATION = "Assert";

	static boolean isTestImport(IImportDeclaration imp) {
		String name = imp.getElementName();
		return (name.startsWith(MAGICTEST_ANNOTATION_PKG) || name.startsWith(TESTNG_ANNOTATION_PKG) || name.startsWith(JUNIT_ANNOTATION_PKG));
	}

	static boolean isTestAnnotation(IAnnotation annot) {
		String name = annot.getElementName();
		return (name.startsWith(MAGICTEST_ANNOTATION_PKG) || name.startsWith(TESTNG_ANNOTATION_PKG) || name.startsWith(JUNIT_ANNOTATION_PKG)
				|| name.equals(TRACE_ANNOTATION) || name.equals(CAPTURE_ANNOTATION) || name.equals(TEST_ANNOTATION) || name.equals(ASSERT_ANNOTATION));
	}

	/**
	 * Find a test class for the specified project and class and return it.
	 *
	 * @param testProjectName   name of test project
	 * @param className         name of class
	 * @return                  found test class or null
	 */
	static IType getType(String testProjectName, String className) {
		IType type = JdtTools.getType(testProjectName, className);
		if (type != null) {
			return type;
		}
		return null;
	}

	static IType getTestType(String prjName, String className) {
		String packageName = ClassTools.getParentNameByDot(className);
		className = ClassTools.getLocalNameByDot(className);

		String testClassName = ClassTools.getFullNameByDot(packageName, "Test" + className);
		IType type = JdtTools.getType(prjName, testClassName);
		if (type != null) {
			return type;
		}
		testClassName = ClassTools.getFullNameByDot(packageName, className + "Test");
		type = JdtTools.getType(prjName, testClassName);
		if (type != null) {
			return type;
		}
		return null;
	}

	static IType getTesteeType(String prjName, String className) {
		String pkgName = ClassTools.getParentNameByDot(className);
		className = ClassTools.getLocalNameByDot(className);

		className = StringTools.removeHeadIf(className, "Test");
		className = StringTools.removeTailIf(className, "Test");

		className = ClassTools.getFullNameByDot(pkgName, className);
		IType type = JdtTools.getType(prjName, className);
		if (type != null) {
			return type;
		}
		return null;
	}

	/**
	 * Show actual report for current selection.
	 */
	public void showActualReport() {
		SelectInfo selectInfo = getSelectInfo();
		LOG.debug("CommandHandler.showActualReport:" + selectInfo);
		if (selectInfo != null) {
			BrowserSupport.showActualReport(selectInfo);
		}
	}

	/**
	 * Show reference report for current selection.
	 */
	public void showReferenceReport() {
		SelectInfo selectInfo = getSelectInfo();
		LOG.debug("CommandHandler.showReferenceReport:" + selectInfo);
		if (selectInfo != null) {
			BrowserSupport.showReferenceReport(selectInfo);
		}
	}

	/**
	 * Returns info about current selection.
	 *
	 * @return info about current selection
	 */
	SelectInfo getSelectInfo() {
		SelectInfo selectInfo = getSelectInfo(event);
		if (selectInfo == null || selectInfo.projectName == null
				|| (selectInfo.className == null && selectInfo.packageName == null && selectInfo.sourceFolder == null)) {
			// MagicTestPlugin.showInfo("Select class or method for executing this command");
			return null;
		}
		return selectInfo;
	}

	/**
	 * Returns information about the selected Java element in the Eclipse
	 * workbench. The information is retrieved from the Java package explorer,
	 * the Java editor or the Java editor outline.
	 *
	 * @return select information
	 */
	static SelectInfo getSelectInfo(ExecutionEvent event) {
		String id = HandlerUtil.getActivePartId(event);
		if (id.equals("org.eclipse.jdt.ui.CompilationUnitEditor")) {
			return getJavaEditorInfo(event);
		} else if (id.equals("org.eclipse.ui.views.ContentOutline")) {
			return getJavaEditorInfo(event);
		} else if (id.equals("org.eclipse.jdt.ui.PackageExplorer") || id.equals("org.eclipse.ui.navigator.ProjectExplorer")) {
			return getExplorerInfo(event);
		} else {
			return null;
		}
	}

	/**
	 * Returns information about the selected Java element in the Project or Package
	 * explorer.
	 *
	 * @param event execution event
	 * @return      select information
	 */
	static SelectInfo getExplorerInfo(ExecutionEvent event) {
		try {
			ISelection sel = HandlerUtil.getActiveMenuSelection(event);
			IStructuredSelection selection = (IStructuredSelection) sel;
			Object firstElement = selection.getFirstElement();
			IJavaElement node = ((IJavaElement) firstElement);

			// TODO: Fails with duplicate class definition error...
			if (node instanceof IPackageFragmentRoot) {
				return getSourceFolder((IPackageFragmentRoot) node);
			}

			// IMethod: run
			// IType: Main
			// ICompilationUnit: Main.java
			// IPackageFragment: top.pkg
			SelectInfo selectInfo = new SelectInfo();
			while (node != null) {
				// System.out.println(node.getClass().getName() + ": " + node);
				if (node instanceof IJavaProject) {
					// Java project
					IJavaProject prj = (IJavaProject) node;
					selectInfo.projectName = prj.getElementName();
				} else if (node instanceof IPackageFragmentRoot) {
					// Source folder
					IPackageFragmentRoot root = (IPackageFragmentRoot) node;
					for (IJavaElement child : root.getChildren()) {
						if (child instanceof IPackageFragment) {
							IPackageFragment pkg = (IPackageFragment) child;
							//System.out.println(pkg.getElementName());
						}
					}
				} else if (node instanceof IPackageFragment) {
					// Java package
					IPackageFragment pkg = (IPackageFragment) node;
					if (selectInfo.className != null) {
						selectInfo.className = pkg.getElementName() + "." + selectInfo.className;
					} else {
						selectInfo.packageName = pkg.getElementName();
						ICompilationUnit[] units = pkg.getCompilationUnits();
						for (ICompilationUnit unit : units) {
							String className = null;
							if (unit.getElementName().endsWith(".java")) {
								className = unit.getElementName().substring(0, unit.getElementName().length() - ".java".length());
								//System.out.println(className);
							}
						}
					}
				} else if (node instanceof ICompilationUnit) {
					ICompilationUnit unit = (ICompilationUnit) node;
					if (selectInfo.className == null) {
						if (unit.getElementName().endsWith(".java")) {
							selectInfo.className = unit.getElementName().substring(0, unit.getElementName().length() - ".java".length());
						}
					}
				} else if (node instanceof IType) {
					IType type = (IType) node;
					selectInfo.className = type.getElementName();
				} else if (node instanceof IMethod) {
					IMethod method = (IMethod) node;
					selectInfo.methodName = method.getElementName();
				}
				node = node.getParent();
			}
			return selectInfo;
		} catch (JavaModelException e) {
			throw new RuntimeException(e);
		}
	}

	static SelectInfo getSourceFolder(IPackageFragmentRoot pfr) throws JavaModelException {
		SelectInfo selectInfo = new SelectInfo();
		selectInfo.sourceFolder = pfr.getElementName();
		IJavaProject prj = pfr.getJavaProject();
		selectInfo.projectName = prj.getElementName();
		return selectInfo;
	}

	/**
	 * Returns information about the selected Java element in the Java editor or
	 * Java editor outline.
	 *
	 * @param event
	 *            execution event
	 * @return select information
	 */
	static SelectInfo getJavaEditorInfo(ExecutionEvent event) {
		JavaEditor editor = (JavaEditor) HandlerUtil.getActiveEditor(event);

		SelectInfo ri = new SelectInfo();
		IEditorInput input = editor.getEditorInput();
		if (!(input instanceof IFileEditorInput))
			return null;
		IFile file = ((IFileEditorInput) input).getFile();
		ri.projectName = file.getProject().getName();

		// Get source viewer
		ISourceViewer viewer = editor.getViewer();
		if (viewer == null)
			return null;

		// Get the caret position
		Point selectedRange = viewer.getSelectedRange();
		int caretAt = selectedRange.x;
		int length = selectedRange.y;

		// Get the Java element
		ITypeRoot element = JavaUI.getEditorInputTypeRoot(editor.getEditorInput());
		if (element == null)
			return null;

		// Get the compilation unit AST
		CompilationUnit ast = SharedASTProvider.getAST(element, SharedASTProvider.WAIT_YES, null);
		if (ast == null)
			return null;

		// Find the node at caret position
		NodeFinder finder = new NodeFinder(caretAt, length);
		ast.accept(finder);

		ASTNode originalNode = finder.getCoveringNode();
		ASTNode node;

		// Dump node hierarchy
		node = originalNode;
		while (node != null) {
			// System.out.println(node.getClass().getName() + ": " + node);
			if (node instanceof MethodDeclaration) {
				MethodDeclaration method = (MethodDeclaration) node;
				String methodName = method.getName().getIdentifier();
				ri.methodName = methodName;
				// System.out.println("Method: " + methodName);
			} else if (node instanceof TypeDeclaration) {
				TypeDeclaration type = (TypeDeclaration) node;
				// type.isInterface()
				String typeName = type.getName().getIdentifier();
				ri.className = typeName;
				// System.out.println("Class: " + typeName);
			} else if (node instanceof CompilationUnit) {
				CompilationUnit unit = (CompilationUnit) node;
				if (unit.getPackage() == null) {
					// default package
					ri.className = ri.className;
				} else {
					String pkgName = unit.getPackage().getName().getFullyQualifiedName();
					ri.className = pkgName + "." + ri.className;
				}
				// System.out.println("Package: " + pkgName);
			}
			node = node.getParent();
		}
		return ri;
	}

}
