package org.magictest.eclipse;

import static org.magictest.MagicTest.Args.*;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.Launch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.IVMRunner;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.magictest.MagicTest;
import org.magictest.eclipse.tools.JdtTools;
import org.magictest.ext.ch.qos.logback.classic.Logger;
import org.magictest.ext.org.magicwerk.brownies.core.ThreadTools;
import org.magictest.ext.org.slf4j.LoggerFactory;
import org.magictest.testng.eclipse.MagicTestPlugin;
import org.magictest.testng.eclipse.VMRunnerTools;
import org.magictest.testng.eclipse.util.PreferenceStoreUtil;

/**
 * Run test using MagicTest.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class MagicTestLauncher {
	/** Logger */
	private static Logger LOG = (Logger) LoggerFactory.getLogger(MagicTestLauncher.class);

	/**
	 * Confirm result of test.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void saveTest(SelectInfo selectInfo) {
		LOG.debug("saveTest: " + selectInfo.projectName);
		IJavaProject prj = JdtTools.getJavaProject(selectInfo.projectName);
		PreferenceStoreUtil storage = MagicTestPlugin.getPluginPreferenceStore();
		String dir = storage.getMagicTestOutPathAbs(prj).toOSString();

		String[] args;
		if (selectInfo.packageName != null) {
			args = new String[] { SAVE, DIR, dir, PACKAGE, selectInfo.packageName };
		} else if (selectInfo.methodName != null) {
			args = new String[] { SAVE, DIR, dir, METHOD, selectInfo.className + "." + selectInfo.methodName };
		} else {
			args = new String[] { SAVE, DIR, dir, CLASS, selectInfo.className };
		}

		// The command must be launched as independent process because the
		// test class is not available on the class path
		launch(prj, MagicTest.MAIN_CLASS, args);
	}

	/**
	 * Delete test.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void deleteTest(SelectInfo selectInfo) {
		LOG.debug("deleteTest: " + selectInfo.projectName);
		IJavaProject prj = JdtTools.getJavaProject(selectInfo.projectName);
		PreferenceStoreUtil storage = MagicTestPlugin.getPluginPreferenceStore();
		String dir = storage.getMagicTestOutPathAbs(prj).toOSString();

		String[] args;
		if (selectInfo.packageName != null) {
			args = new String[] { DELETE, DIR, dir, PACKAGE, selectInfo.packageName };
		} else if (selectInfo.methodName != null) {
			if (selectInfo.methodName.equals(MISSING)) {
				args = new String[] { DELETE, DIR, dir, MISSING, selectInfo.className };
			} else {
				args = new String[] { DELETE, DIR, dir, METHOD, selectInfo.className + "." + selectInfo.methodName };
			}
		} else {
			args = new String[] { DELETE, DIR, dir, CLASS, selectInfo.className };
		}

		// The command must be launched as independent process because the
		// test class is not available on the class path
		launch(prj, MagicTest.MAIN_CLASS, args);
	}

	/**
	 * Invalidate test.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void invalidateTest(SelectInfo selectInfo) {
		LOG.debug("invalidateTest: " + selectInfo.projectName);
		IJavaProject prj = JdtTools.getJavaProject(selectInfo.projectName);
		PreferenceStoreUtil storage = MagicTestPlugin.getPluginPreferenceStore();
		String dir = storage.getMagicTestOutPathAbs(prj).toOSString();

		String[] args;
		if (selectInfo.packageName != null) {
			args = new String[] { INVALIDATE, DIR, dir, PACKAGE, selectInfo.packageName };
		} else if (selectInfo.methodName != null) {
			args = new String[] { INVALIDATE, DIR, dir, METHOD, selectInfo.className + "." + selectInfo.methodName };
		} else {
			args = new String[] { INVALIDATE, DIR, dir, CLASS, selectInfo.className };
		}

		// The command must be launched as independent process because the
		// test class is not available on the class path
		launch(prj, MagicTest.MAIN_CLASS, args);
	}

	/**
	 * Launches the given program. After the launch, the method waits until
	 * the program has finished, then writes its output to the console and
	 * finally returns.
	 *
	 * @param proj project where program to launch resides in
	 * @param main name of main class to execute
	 * @param args command line arguments to pass to the program
	 */
	// http://wiki.eclipse.org/FAQ_How_do_I_launch_a_Java_program%3F
	static void launch(IJavaProject proj, String main, String[] args) {
		try {
			IVMInstall vm = JavaRuntime.getVMInstall(proj);
			if (vm == null) {
				vm = JavaRuntime.getDefaultVMInstall();
			}
			IVMRunner vmr = vm.getVMRunner(ILaunchManager.RUN_MODE);
			String[] cp = JavaRuntime.computeDefaultRuntimeClassPath(proj);
			VMRunnerConfiguration config = new VMRunnerConfiguration(main, cp);
			config.setProgramArguments(args);
			LOG.debug("MagicTestLauncher: " + VMRunnerTools.toString(config));

			ILaunch launch = new Launch(null, ILaunchManager.RUN_MODE, null);
			//workingCopy.setAttribute(IExternalToolConstants.ATTR_CAPTURE_OUTPUT, true);
			//launch.setAttribute(DebugPlugin.ATTR_CAPTURE_OUTPUT, "true");
			vmr.run(config, launch, null);
			IProcess processes[] = launch.getProcesses();

			// Wait until all processes are terminated
			// TODO: no spinning
			while (true) {
				int i;
				for (i = 0; i < processes.length; i++) {
					IProcess process = processes[i];
					if (!process.isTerminated()) {
						break;
					}
				}
				if (i == processes.length) {
					break;
				}
				// Wait some time
				ThreadTools.sleep(100);
			}

			// Read output from terminated processes
			for (int i = 0; i < processes.length; i++) {
				IProcess process = processes[i];
				String out = process.getStreamsProxy().getOutputStreamMonitor().getContents();
				if (out != null && out.trim().length() > 0) {
					LOG.info(out);
				}
				String err = process.getStreamsProxy().getErrorStreamMonitor().getContents();
				if (err != null && err.trim().length() > 0) {
					LOG.info(err);
				}
			}
		} catch (CoreException e) {
			throw new RuntimeException(e);
		}
	}
}
