package org.magictest.eclipse;

import java.net.URI;
import java.net.URL;
import java.util.Arrays;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.swt.browser.Browser;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.browser.IWebBrowser;
import org.eclipse.ui.browser.IWorkbenchBrowserSupport;
import org.eclipse.ui.internal.browser.BrowserViewer;
import org.eclipse.ui.internal.browser.WebBrowserEditor;
import org.eclipse.ui.internal.browser.WebBrowserEditorInput;
import org.magictest.HtmlTestWriter.Command;
import org.magictest.MagicTest.Args;
import org.magictest.MagicTestTools;
import org.magictest.TestReporter;
import org.magictest.eclipse.tools.JdtTools;
import org.magictest.eclipse.tools.ResourceTools;
import org.magictest.ext.org.magicwerk.brownies.core.ObjectTools;
import org.magictest.testng.eclipse.MagicTestPlugin;
import org.magictest.testng.eclipse.util.PreferenceStoreUtil;
import org.magictest.testng.eclipse.util.SWTUtil;
import org.magicwerk.brownies.core.StringTools;
import org.magicwerk.brownies.core.collections.CollectionQueries;
import org.magicwerk.brownies.core.collections.Iterate;
import org.magicwerk.brownies.core.files.FilePath;
import org.magicwerk.brownies.core.files.FilePath.ParentMode;
import org.magicwerk.brownies.core.objects.Tuple;
import org.magicwerk.brownies.core.reflect.ClassTools;
import org.magicwerk.brownies.core.reflect.ReflectTools;
import org.magicwerk.brownies.swt.BrowserTools;
import org.magicwerk.brownies.swt.LinkListener;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Logger;

/**
 * Supporting methods for the Eclipse browser.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class BrowserSupport {
	/*
	 * There seems to be an issue with the integrated Eclipse browser that it does not refresh a page
	 * with a fragment, if the URL does not change. One solution to force refresh would be to add
	 * a unique invocation count to the URL, like "testClass.act.html?1#testMethod". This causes
	 * however disturbing flickering as the page is loaded several times, once when the test run starts
	 * (LaunchUtil.showReport) and then after the completion of each test method (TestRunnerViewPart.postTestResult).
	 * Therefore another approach has been taken: on start of the test run, the top of the page is shown
	 * without fragments. If a whole class or package is tested, the page view will remain unchanged, so
	 * the test overview is visible. If a single method is tested, the result of the test run is shown
	 * using a fragment "testClass.act.html#testMethod" if the test has been completed.
	 */
	/** Logger */
	private static final Logger LOG = (Logger) LoggerFactory.getLogger(BrowserSupport.class);
	/** Editor ID of browser */
	private static final String BROWSER_ID = "org.eclipse.ui.browser.editor";

	/** Information about current MagicTest run started */
	private static SelectInfo currRun;
	/** True if current run was started in profile mode, i.e. report will not be updated */
	private static boolean currProfile;
	/** Counter which is incremented each time a URL with anchor is shown */
	private static int anchorCount;

	/**
	 * Show HTML file.
	 *
	 * @param file		file to show
	 * @param anchor	anchor to jump to or null to show top of page
	 */
	static void showHtmlFile(IFile file, String anchor) {
		try {
			URI uri = file.getLocationURI();
			URL url = uri.toURL();

			String browserId = url.toString();
			String name = browserId;
			String tooltip = browserId;
			IWorkbenchBrowserSupport browserSupport = PlatformUI.getWorkbench().getBrowserSupport();
			IWebBrowser browser = browserSupport.createBrowser(
					IWorkbenchBrowserSupport.AS_EDITOR | IWorkbenchBrowserSupport.LOCATION_BAR | IWorkbenchBrowserSupport.NAVIGATION_BAR, browserId, name,
					tooltip);
			if (anchor != null) {
				// Workaround: The internal browser (IE with Windows 7/8) used by Eclipse Indigo 3.7.1
				// does not refresh the HTML page if browse.openURL() is called again with the
				// same URL which includes an anchor. If there is no anchor, refresh work.
				// As a workaround, we add a distinct dummy query string argument to the URL.
				url = new URL(url.toString() + "?" + anchorCount + "#" + anchor);
				anchorCount++;
			}
			browser.openURL(url);
		} catch (Exception ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Checks whether the specified file is open in a browser.
	 *
	 * @param file absolute file path of HTML file
	 * @return     true if HTML file is open in an editor (whether active or not)
	 */
	static boolean isHtmlFileOpen(String file) {
		IEditorReference[] editors = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getEditorReferences();
		for (IEditorReference editor : editors) {
			if (editor.getId().equals(BROWSER_ID)) {
				if (ObjectTools.equals(FilePath.of(editor.getTitle()), FilePath.of(file))) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Checks whether the specified file is open in a browse which is the active window.
	 *
	 * @param file	absolute file path of HTML file
	 * @return      true if HTML file is open in the active editor
	 */
	static boolean isHtmlFileActive(String file) {
		IEditorPart part = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActiveEditor();
		if (part instanceof WebBrowserEditor) {
			WebBrowserEditor editor = (WebBrowserEditor) part;
			if (ObjectTools.equals(FilePath.of(editor.getTitle()), FilePath.of(file))) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Refresh all open HTML files which are specified.
	 *
	 * @param selectInfo
	 */
	public static void refreshOpenHtmlFiles(SelectInfo selectInfo) {
		try {
			IEditorReference[] editors = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getEditorReferences();
			for (IEditorReference editor : editors) {
				IEditorPart part = (IEditorPart) editor.getPart(false);
				SelectInfo browserSelInfo = getInfoFromPart(part);
				if (browserSelInfo != null) {
					if (browserSelInfo.projectName.equals(selectInfo.projectName)) {
						if ((selectInfo.packageName != null && browserSelInfo.className.startsWith(selectInfo.packageName))
								|| selectInfo.packageName == null && browserSelInfo.className.equals(selectInfo.className)) {
							LOG.debug("Refreshing active HTML browser");
							WebBrowserEditor webEditor = (WebBrowserEditor) part;
							BrowserViewer browserViewer = (BrowserViewer) ReflectTools.getAnyBeanValue(webEditor, "webBrowser");
							browserViewer.refresh();
						}
					}
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Refreshes content in HTML browser if active.
	 */
	static void refreshActiveHtmlFile() {
		IEditorPart part = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActiveEditor();
		SelectInfo selInfo = getInfoFromPart(part);
		if (selInfo != null) {
			LOG.debug("Refreshing active HTML browser");
			WebBrowserEditor webEditor = (WebBrowserEditor) part;
			BrowserViewer browserViewer = (BrowserViewer) ReflectTools.getAnyBeanValue(webEditor, "webBrowser");
			browserViewer.refresh();
		}
	}

	/**
	 * Show actual test report in browser.
	 * The report is shown in any case.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void showReportOnRun(SelectInfo selectInfo, boolean profile) {
		if (!profile) {
			String method = selectInfo.methodName;
			selectInfo.methodName = null;
			showTestReport(selectInfo, false, true);
			selectInfo.methodName = method;
		}
		currRun = selectInfo;
		currProfile = profile;
	}

	/**
	 * Show actual test report in browser.
	 * The report is shown if it contains errors or it is active.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void showReportAfterRun(SelectInfo selectInfo) {
		// If a whole test class is run, we do not jump to the methods
		if (currRun == null || currRun.methodName == null) {
			selectInfo.methodName = null;
		}
		if (!currProfile) {
			showTestReport(selectInfo, false, false);
		}
		currProfile = false;
	}

	/**
	 * Show actual test report in browser.
	 * The report is shown in any case.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void showActualReport(SelectInfo selectInfo) {
		showTestReport(selectInfo, false, true);
	}

	/**
	 * Show actual test report in browser.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void showReferenceReport(SelectInfo selectInfo) {
		showTestReport(selectInfo, true, true);
	}

	/**
	 * Returns MagicTest output directory for project.
	 *
	 * @param projName  project name
	 * @return			MagicTest output directory
	 */
	static String getMagicTestDir(String projName) {
		PreferenceStoreUtil storage = MagicTestPlugin.getPluginPreferenceStore();
		String outDir = storage.getProjectProperties(projName).getMagicTestDir();
		return outDir;
	}

	/**
	 * Show test report in browser.
	 *
	 * @param selectInfo information about the selection in the user interface
	 * @param ref true to show reference result, false for actual result
	 * @param always true to show HTML always, false to show it only on error
	 */
	static void showTestReport(SelectInfo selectInfo, boolean ref, boolean always) {
		// Determine name of HTML file
		String outDir = getMagicTestDir(selectInfo.projectName);
		String resultFile = TestReporter.getFile(selectInfo.className, outDir, ref, true);
		LOG.info("Show HTML file {}", resultFile);

		// As the file has been modified outside Eclipse, we have to manually refresh it
		IFile file = ResourceTools.getFile(selectInfo.projectName, resultFile);
		try {
			file.refreshLocal(IFile.DEPTH_ZERO, null);
		} catch (CoreException e) {
			LOG.warn("Refreshing " + resultFile + " failed", e);
		}
		if (!file.exists()) {
			return;
		}
		if (!always) {
			String path = ResourceTools.getFileLocation(file);
			if (!isHtmlFileOpen(path)) {
				// Show not active file only if it contains errors
				PreferenceStoreUtil storage = MagicTestPlugin.getPluginPreferenceStore();
				IJavaProject prj = JdtTools.getJavaProject(selectInfo.projectName);
				String dir = storage.getMagicTestOutPathAbs(prj).toOSString();
				if (MagicTestTools.isSuccess(selectInfo.className, dir)) {
					return;
				}
			}
		}

		String anchor = selectInfo.methodName;
		if (anchor != null) {
			if (anchor.equals(Args.MISSING)) {
				anchor = null;
			}
		}
		showHtmlFile(file, anchor);
	}

	/**
	 * Refresh information about files which have been modified outside Eclipse.
	 *
	 * @param selectInfo information about the selection in the user interface
	 */
	public static void refreshMagicTestDir(SelectInfo selectInfo) {
		String outDir = getMagicTestDir(selectInfo.projectName);
		IFolder folder = ResourceTools.getFolder(selectInfo.projectName, outDir);
		try {
			folder.refreshLocal(IFile.DEPTH_INFINITE, null);
		} catch (CoreException e) {
			LOG.warn("Refreshing " + outDir + " failed", e);
		}
	}

	/**
	 * Handle event if user clicked on a browser link.
	 *
	 * @param projectName name of project
	 * @param command command which has been sent by clicking the browser link
	 */
	static void handleBrowserLink(String projectName, String command) {
		LOG.debug("handleBrowserLink: {}, {}", projectName, command);

		Command cmd = null;
		for (Command c : Command.values()) {
			if (command.startsWith(c.getCommandPrefix())) {
				command = command.substring(c.getCommandPrefix().length());
				cmd = c;
				break;
			}
		}
		if (cmd == null) {
			LOG.debug("handleBrowserLink: invalid link {}, {}", projectName, command);
			return;
		}

		SelectInfo selectInfo = new SelectInfo();
		selectInfo.projectName = projectName;
		if (command.startsWith("class-")) {
			command = command.substring("class-".length());
			selectInfo.className = command;
		} else if (command.startsWith("method-")) {
			command = command.substring("method-".length());
			int index = command.lastIndexOf('.');
			selectInfo.className = command.substring(0, index);
			selectInfo.methodName = command.substring(index + 1);
		}

		if (cmd == Command.RUN) {
			LOG.debug("Running test " + selectInfo);
			CommandHandler.runTest(selectInfo);
		} else if (cmd == Command.SAVE) {
			LOG.debug("Saving test " + selectInfo);
			CommandHandler.saveTest(selectInfo);
		} else if (cmd == Command.DELETE) {
			LOG.debug("Delete test " + selectInfo);
			CommandHandler.deleteTest(selectInfo);
		} else if (cmd == Command.INVALIDATE) {
			LOG.debug("Invalidate test " + selectInfo);
			CommandHandler.invalidateTest(selectInfo);
		} else if (cmd == Command.SHOW) {
			LOG.debug("Show test " + selectInfo);
			CommandHandler.showTest(selectInfo, false);
		} else {
			assert (false);
		}
	}

	/**
	 * Watch all browser windows for clicks on links
	 */
	public static void watchBrowser() {
		SWTUtil.getDisplay().asyncExec(new Runnable() {
			@Override
			public void run() {
				doWatchBrowser();
			}
		});
	}

	/**
	 * Returns browser viewer for a MagicTest report.
	 *
	 * @param part workbench part
	 * @return     browser viewer of a MagicTest report or null
	 */
	static Tuple<BrowserViewer, SelectInfo> getBrowserViewer(IWorkbenchPart part) {
		SelectInfo selInfo = getInfoFromPart(part);
		if (selInfo != null) {
			// TODO: internal access
			WebBrowserEditor webEditor = (WebBrowserEditor) part;
			BrowserViewer browserViewer = (BrowserViewer) ReflectTools.getAnyBeanValue(webEditor, "webBrowser");
			return new Tuple<BrowserViewer, SelectInfo>(browserViewer, selInfo);
		}
		return null;
	}

	public static void doWatchBrowser() {
		LOG.info("Watching browser for HTML reports");
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		final IWorkbenchPage page = window.getActivePage();
		page.addPartListener(new IPartListener() {
			@Override
			public void partOpened(IWorkbenchPart part) {
				LOG.debug("partOpened {}", part);
				Tuple<BrowserViewer, SelectInfo> info = getBrowserViewer(part);
				if (info != null) {
					BrowserViewer browserViewer = info.getItem0();
					final SelectInfo selInfo = info.getItem1();
					LOG.info("Activating browser links for {}", selInfo.className);
					Browser browser = browserViewer.getBrowser();
					BrowserTools.addLinkListener(browser, new LinkListener() {
						@Override
						public void clicked(String command) {
							handleBrowserLink(selInfo.projectName, command);
						}
					});
				}
			}

			@Override
			public void partClosed(IWorkbenchPart part) {
			}

			@Override
			public void partActivated(IWorkbenchPart part) {
			}

			@Override
			public void partBroughtToTop(IWorkbenchPart part) {
			}

			@Override
			public void partDeactivated(IWorkbenchPart part) {
			}
		});
	}

	static SelectInfo getInfoFromPart(IWorkbenchPart part) {
		if (part instanceof WebBrowserEditor) {
			WebBrowserEditor webEditor = (WebBrowserEditor) part;
			IEditorInput inp = webEditor.getEditorInput();
			if (inp instanceof WebBrowserEditorInput) {
				WebBrowserEditorInput wbei = (WebBrowserEditorInput) inp;
				URL url = wbei.getURL();
				FilePath path = FilePath.of(url.toString());
				SelectInfo selInfo = getInfoFromHtmlReport(path.getPath());
				return selInfo;
			}
		}
		return null;
	}

	/**
	 * Checks whether the given file can be a HTML report file.
	 *
	 * @param file path of HTML file which is opened (form D:/Java/Sources/Origo-Magicwerk/MagicTest-Test/magictest/org/magictest/presentation/StringUtilsTest.act.html)
	 * @return SelectInfo containing projectName and className if file is a HTML report file, null otherwise
	 */
	static SelectInfo getInfoFromHtmlReport(String file) {
		LOG.info("getInfoFromHtmlReport: {}", file);

		// Remove anchor part if it exists
		int pos = file.indexOf("#");
		if (pos != -1) {
			file = file.substring(0, pos);
		}

		// Look for MagicTest file ending and remove it
		String file2 = null;
		if (file2 == null) {
			file2 = StringTools.removeTail(file, TestReporter.ACT_HTML_SUFFIX);
		}
		if (file2 == null) {
			file2 = StringTools.removeTail(file, TestReporter.REF_HTML_SUFFIX);
		}
		if (file2 == null) {
			return null;
		}
		file = file2;

		FilePath p = FilePath.of(file);
		return getInfoFromFile(p);
	}

	static SelectInfo getInfoFromFile(FilePath file) {
		LOG.debug("getInfoFromFile.file: {}", file);
		IJavaProject jp = getProjectFromFile(file);
		if (jp == null) {
			return null;
		}

		SelectInfo selectInfo = new SelectInfo();
		selectInfo.projectName = jp.getProject().getName();
		LOG.debug("getInfoFromFile.selectInfo.projectName: {}", selectInfo.projectName);

		String outDir = getMagicTestDir(selectInfo.projectName);
		LOG.debug("getInfoFromFile.outDir: {}", outDir);
		FilePath dir = getPathFromProject(jp).get(outDir);
		LOG.debug("getInfoFromFile.dir: {}", dir);
		FilePath classFile = dir.relativize(file, ParentMode.NULL);
		if (classFile == null) {
			return null;
		}

		selectInfo.className = ClassTools.getClassFromPath(classFile.getPath());
		LOG.debug("getInfoFromFile.selectInfo.className: {}", selectInfo.className);
		return selectInfo;
	}

	static IJavaProject getProjectFromFile(FilePath file) {
		// If we have file /dir1/dir2/file and 2 projects in /dir1 and /dir1/dir2, make sure we select the one in /dir1/dir2
		return CollectionQueries.getMinBy(Iterate.of(Arrays.asList(JdtTools.getJavaProjects())), (IJavaProject jp) -> {
			FilePath prjDir = getPathFromProject(jp);
			FilePath d = prjDir.relativize(file, ParentMode.NULL);
			if (d == null) {
				return null;
			}
			return d.getPath().length();
		});
	}

	static FilePath getPathFromProject(IJavaProject jp) {
		return FilePath.of(jp.getProject().getLocation().toString());
	}

}
