package org.magictest.eclipse.tools;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IAnnotation;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.texteditor.ITextEditor;
import org.magictest.TestInstrumentator;

/**
 * Tools for working with the JDT.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class JdtTools {

	// Java Model

	public static IJavaModel getJavaModel() {
		return JavaCore.create(ResourceTools.getWorkspaceRoot());
	}

	// Editor

	/**
	 * Open and activate an editor for specified element.
	 * This method works for element types IType and IMethod.
	 *
	 * @param elem  element to open and activate
	 */
	public static void open(IJavaElement elem) {
		IJavaElement type;
		if (elem instanceof IMethod) {
			type = ((IMethod) elem).getParent();
		} else {
			type = elem;
		}
		try {
			ITextEditor textEditor = (ITextEditor) EditorUtility.openInEditor(type, true);
			if (elem instanceof IMember) {
				IMember member = (IMember) elem;
				ISourceRange range = member.getNameRange();
				textEditor.selectAndReveal(range.getOffset(), range.getLength());
			}
		} catch (PartInitException e) {
			//throw new RuntimeException(e);
		} catch (JavaModelException e) {
			//throw new RuntimeException(e);
		}
	}

	public static void reveal(IMember element) {
	}

	// Projects

	/**
	 * Returns Java project.
	 * Note that the name may also reference a closed or
	 * even a non-existing project.
	 *
	 * @param projectName	project name
	 * @return				project
	 */
	public static IJavaProject getJavaProject(String projectName) {
		return getJavaModel().getJavaProject(projectName);
	}

	public static IJavaProject[] getJavaProjects() {
		try {
			return getJavaModel().getJavaProjects();
		} catch (JavaModelException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Returns source folders.
	 * If a source folder is linked, the linked folder location is included.
	 *
	 * @param javaProject   Java project
	 * @return              list with source folders
	 */
	public static List<String> getSourceFolders(IJavaProject javaProject) {
		List<String> sourcePaths = new ArrayList<String>();
		try {
			IPackageFragmentRoot[] sourceRoots = javaProject.getAllPackageFragmentRoots();
			for (IPackageFragmentRoot sourceRoot : sourceRoots) {
				if (sourceRoot.getKind() == IPackageFragmentRoot.K_SOURCE) {
					IResource sourceRes = sourceRoot.getCorrespondingResource();
					if (null != sourceRes) {
						sourcePaths.add(sourceRes.getLocation().toOSString());
					}
				}
			}
		} catch (JavaModelException e) {
			throw new RuntimeException(e);
		}
		return sourcePaths;
	}

	// Java internals

	public static IPackageFragment getPackage(String projectName, String packageName) {
		IJavaProject javaProject = getJavaProject(projectName);
		if (javaProject == null) {
			return null;
		}
		return getPackage(javaProject, packageName);
	}

	public static IPackageFragmentRoot getSourceFolder(String projectName, String sourceFolder) {
		IJavaProject javaProject = getJavaProject(projectName);
		if (javaProject == null) {
			return null;
		}
		return getSourceFolder(javaProject, sourceFolder);
	}

	public static IType getType(String projectName, String className) {
		IJavaProject javaProject = getJavaProject(projectName);
		if (javaProject == null) {
			return null;
		}
		return getType(javaProject, className);
	}

	/**
	 * Get method with specified project, class and method name.
	 */
	public static IMethod getMethod(String projectName, String className, String methodName) {
		IType type = getType(projectName, className);
		if (type == null) {
			return null;
		}
		return getMethod(type, methodName);
	}

	/**
	 * Get test method with specified project, class and method name.
	 * If there is no test method in the class, search continues in the super class.
	 * A test method has MagicTest annotation.
	 */
	public static IMethod getTestMethod(String projectName, String className, String methodName) {
		IType type = getType(projectName, className);
		return getTestMethod(type, methodName);
	}

	public static IMethod getTestMethod(IType type, String methodName) {
		while (type != null) {
			IMethod method = getMethod(type, methodName);
			if (method != null) {
				return method;
			}

			type = getSuperclass(type);
		}
		return null;
	}

	public static IType getSuperclass(IType type) {
		try {
			return type.newSupertypeHierarchy(null).getSuperclass(type);
			//return getType(javaProject, type.getSuperclassName());
		} catch (CoreException e) {
			return null;
		}
	}

	public static boolean isTestMethod(IMethod method) {
		IAnnotation[] annos = getAnnotations(method);
		for (IAnnotation anno : annos) {
			String clazz = anno.getClass().toString();
			if (TestInstrumentator.isMagicTestAnnotation(clazz)) {
				return true;
			}
		}
		return false;
	}

	public static IAnnotation[] getAnnotations(IMethod method) {
		try {
			return method.getAnnotations();
		} catch (JavaModelException e) {
			throw new RuntimeException(e);
		}
	}

	public static IPackageFragment getPackage(IJavaProject javaProject, String packageName) {
		try {
			IPackageFragment[] pkgs = javaProject.getPackageFragments();
			for (IPackageFragment pkg : pkgs) {
				if (pkg.getElementName().equals(packageName)) {
					return pkg;
				}
			}
			return null;
		} catch (JavaModelException e) {
			return null;
		}
	}

	public static boolean isPackageEmpty(IPackageFragment pkg) {
		try {
			return !pkg.hasChildren();
		} catch (JavaModelException e) {
			throw new RuntimeException(e);
		}
	}

	public static IPackageFragmentRoot getSourceFolder(IJavaProject javaProject, String sourceFolder) {
		try {
			IPackageFragmentRoot[] pkgs = javaProject.getPackageFragmentRoots();
			for (IPackageFragmentRoot pkg : pkgs) {
				if (pkg.getElementName().equals(sourceFolder)) {
					return pkg;
				}
			}
			return null;
		} catch (JavaModelException e) {
			return null;
		}
	}

	public static IType getType(IJavaProject javaProject, String className) {
		try {
			IType type = javaProject.findType(className);
			return type;
		} catch (JavaModelException e) {
			return null;
		}
	}

	public static IMethod getMethod(IType type, String methodName) {
		try {
			IMethod[] methods = type.getMethods();
			for (int i = 0; i < methods.length; i++) {
				if (methods[i].getElementName().equals(methodName)) {
					return methods[i];
				}
			}
			return null;
		} catch (JavaModelException e) {
			return null;
		}
	}

	public static ICompilationUnit getCompilationUnit(IJavaElement je) {
		return (ICompilationUnit) je.getAncestor(IJavaElement.COMPILATION_UNIT);
	}

	public static IJavaElement getJavaElement(IFile file) {
		IJavaElement ie = JavaCore.create(file);
		return ie;
	}

}
