package org.magictest.eclipse.tools;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;

/**
 * Tools for working with IFile objects.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class ResourceTools {

    // Location

    /**
     * Returns location as OS path.
     *
     * @param path
     * @return
     */
    public static String getLocation(IPath path) {
        return path.toOSString();
    }

    // Workspace

    public static IWorkspaceRoot getWorkspaceRoot() {
        return ResourcesPlugin.getWorkspace().getRoot();
    }

    /**
     * Returns location of workspace in file system (e.g. "C:\workspace").
     *
     * @return
     */
    public static String getWorkspaceLocation() {
        return getLocation(getWorkspaceRoot().getLocation());
    }

    // Projects

    public static IProject[] getProjects() {
        return getWorkspaceRoot().getProjects();
    }

    public static IProject getProject(String project) {
        return getWorkspaceRoot().getProject(project);
    }

    /**
     * Returns project location as OS path.
     *
     * @param prj	project
     * @return		project location as OS path
     */
    public static String getProjectLocation(IProject prj) {
        return getLocation(prj.getLocation());
    }

    /**
     * Returns project location as OS path.
     *
     * @param prjName 	project name
     * @return			project location as OS path
     */
    public static String getProjectLocation(String prjName) {
        return getProjectLocation(getProject(prjName));
    }

    // Files

    /**
     * Returns location of file in file system
     * (e.g. 'C:\workspace\project\dir\file').
     *
     * @param file
     * @return
     */
    public static String getFileLocation(IFile file) {
        return getLocation(file.getLocation());
    }

    /**
     * Returns location of project in file system (e.g. "C:\workspace\project\dir\file").
     *
     * @param prj
     * @param path
     * @return
     */
    public static String getFileLocation(IProject prj, String path) {
        return getLocation(prj.getFile(path).getLocation());
    }

    /**
     * Note that the file may or may not exist.
     *
     * @param project
     * @param file
     * @return
     */
    public static IFile getFile(String project, String file) {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IWorkspaceRoot root = workspace.getRoot();
        IProject prj = root.getProject(project);
        IFile ifile = prj.getFile(file);
        return ifile;
    }

    /**
     * Note that the folder may or may not exist.
     *
     * @param project
     * @param file
     * @return
     */
    public static IFolder getFolder(String project, String file) {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IWorkspaceRoot root = workspace.getRoot();
        IProject prj = root.getProject(project);
        IFolder ifolder = prj.getFolder(file);
        return ifolder;
    }

    /**
     * This method returns null if the file does not exist.
     *
     * @param project
     * @param file
     * @return
     */
    public static IFile getExistingFile(String project, String file) {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IWorkspaceRoot root = workspace.getRoot();
        IProject prj = root.getProject(project);
        if (!prj.exists()) {
            return null;
        }
        IFile ifile = prj.getFile(file);
        if (!ifile.exists()) {
            return null;
        }
        return ifile;
    }

    public static IFile getFile(String project, String folder, String file) {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IWorkspaceRoot root = workspace.getRoot();
        IProject prj = root.getProject(project);
        if (!prj.exists()) {
            return null;
        }
        IFolder dir = prj.getFolder(folder);
        if (!dir.exists()) {
            return null;
        }
        IFile ifile = dir.getFile(file);
        if (!ifile.exists()) {
            return null;
        }
        return ifile;
    }

}
