/*
 * Decompiled with CFR 0.152.
 */
package org.mapsforge.poi.writer;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.Normalizer;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.TreeMap;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.mapsforge.core.model.BoundingBox;
import org.mapsforge.core.model.LatLong;
import org.mapsforge.poi.storage.PoiCategory;
import org.mapsforge.poi.storage.PoiCategoryFilter;
import org.mapsforge.poi.storage.PoiCategoryManager;
import org.mapsforge.poi.storage.UnknownPoiCategoryException;
import org.mapsforge.poi.storage.WhitelistPoiCategoryFilter;
import org.mapsforge.poi.writer.GeoTagger;
import org.mapsforge.poi.writer.TagMappingResolver;
import org.mapsforge.poi.writer.XMLPoiCategoryManager;
import org.mapsforge.poi.writer.logging.LoggerWrapper;
import org.mapsforge.poi.writer.logging.ProgressManager;
import org.mapsforge.poi.writer.model.PoiWriterConfiguration;
import org.mapsforge.poi.writer.util.ArabicNormalizer;
import org.openstreetmap.osmosis.core.container.v0_6.EntityContainer;
import org.openstreetmap.osmosis.core.domain.v0_6.Entity;
import org.openstreetmap.osmosis.core.domain.v0_6.Node;
import org.openstreetmap.osmosis.core.domain.v0_6.Relation;
import org.openstreetmap.osmosis.core.domain.v0_6.Tag;
import org.openstreetmap.osmosis.core.domain.v0_6.Way;
import org.openstreetmap.osmosis.core.domain.v0_6.WayNode;

public final class PoiWriter {
    private static final Logger LOGGER = LoggerWrapper.getLogger(PoiWriter.class.getName());
    static final int BATCH_LIMIT = 1024;
    private static final int MIN_NODES_POLYGON = 4;
    private static final GeometryFactory GEOMETRY_FACTORY = new GeometryFactory();
    private static final Pattern NAME_LANGUAGE_PATTERN = Pattern.compile("(name)(:)([a-zA-Z]{1,3}(?:[-_][a-zA-Z0-9]{1,8})*)");
    private static final Pattern NAME_NORMALIZE_PATTERN = Pattern.compile("\\p{InCombiningDiacriticalMarks}+");
    final PoiWriterConfiguration configuration;
    private final ProgressManager progressManager;
    private final NumberFormat nfCounts = NumberFormat.getInstance();
    private final PoiCategoryManager categoryManager;
    private final TagMappingResolver tagMappingResolver;
    private final PoiCategoryFilter categoryFilter;
    private GeoTagger geoTagger;
    private long nNodes = 0L;
    private long nWays = 0L;
    private long nRelations = 0L;
    private long poiAdded = 0L;
    Connection conn = null;
    private PreparedStatement pStmtCatMap = null;
    private PreparedStatement pStmtData = null;
    private PreparedStatement pStmtIndex = null;
    private PreparedStatement pStmtNodesC = null;
    private PreparedStatement pStmtNodesR = null;
    private PreparedStatement pStmtWayNodesR = null;

    public static PoiWriter newInstance(PoiWriterConfiguration configuration, ProgressManager progressManager) {
        return new PoiWriter(configuration, progressManager);
    }

    private PoiWriter(PoiWriterConfiguration configuration, ProgressManager progressManager) {
        this.configuration = configuration;
        this.progressManager = progressManager;
        this.nfCounts.setGroupingUsed(true);
        LOGGER.info("Loading categories...");
        this.categoryManager = new XMLPoiCategoryManager(this.configuration.getTagMapping());
        this.tagMappingResolver = new TagMappingResolver(this.configuration.getTagMapping(), this.categoryManager);
        this.categoryFilter = new WhitelistPoiCategoryFilter();
        try {
            this.categoryFilter.addCategory(this.categoryManager.getRootCategory());
        }
        catch (UnknownPoiCategoryException e) {
            LOGGER.warning("Could not add category to filter: " + e.toString());
        }
        LOGGER.info("Adding tag mappings...");
        try {
            this.prepareDatabase();
        }
        catch (ClassNotFoundException | SQLException | UnknownPoiCategoryException e) {
            e.printStackTrace();
        }
        LOGGER.info("Creating POI database...");
        this.progressManager.initProgressBar(0, 0);
        this.progressManager.setMessage("Creating POI database");
        if (this.configuration.isGeoTags()) {
            this.geoTagger = new GeoTagger(this);
        }
    }

    private void commit() throws SQLException {
        LOGGER.info("Committing...");
        this.progressManager.setMessage("Committing...");
        this.pStmtIndex.executeBatch();
        this.pStmtData.executeBatch();
        this.pStmtCatMap.executeBatch();
        if (this.configuration.isGeoTags()) {
            this.geoTagger.commit();
        }
        this.conn.commit();
    }

    public void complete() {
        if (this.configuration.isGeoTags()) {
            this.geoTagger.processBoundaries();
        }
        NumberFormat nfMegabyte = NumberFormat.getInstance();
        nfMegabyte.setMaximumFractionDigits(2);
        try {
            this.commit();
            if (this.configuration.isFilterCategories()) {
                this.filterCategories();
            }
            this.writeMetadata();
            this.conn.close();
            this.postProcess();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        LOGGER.info("Added " + this.nfCounts.format(this.poiAdded) + " POIs.");
        this.progressManager.setMessage("Done.");
        LOGGER.info("Estimated memory consumption: " + nfMegabyte.format((double)(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) / Math.pow(1024.0, 2.0)) + "MB");
    }

    private void filterCategories() throws SQLException {
        LOGGER.info("Filtering categories...");
        PreparedStatement pStmtChildren = this.conn.prepareStatement("SELECT COUNT(*) FROM poi_categories WHERE parent = ?;");
        PreparedStatement pStmtPoi = this.conn.prepareStatement("SELECT COUNT(*) FROM poi_category_map WHERE category = ?;");
        PreparedStatement pStmtDel = this.conn.prepareStatement("DELETE FROM poi_categories WHERE id = ?;");
        Statement stmt = this.conn.createStatement();
        ResultSet rs = stmt.executeQuery("SELECT id FROM poi_categories ORDER BY id;");
        while (rs.next()) {
            long nPoi;
            long nChildren;
            int id = rs.getInt(1);
            pStmtChildren.setInt(1, id);
            ResultSet rsChildren = pStmtChildren.executeQuery();
            if (!rsChildren.next() || (nChildren = rsChildren.getLong(1)) != 0L) continue;
            pStmtPoi.setInt(1, id);
            ResultSet rsPoi = pStmtPoi.executeQuery();
            if (!rsPoi.next() || (nPoi = rsPoi.getLong(1)) != 0L) continue;
            pStmtDel.setInt(1, id);
            pStmtDel.executeUpdate();
        }
    }

    LatLong findNodeByID(long id) {
        try {
            this.pStmtNodesR.setLong(1, id);
            ResultSet rs = this.pStmtNodesR.executeQuery();
            if (rs.next()) {
                double lat = rs.getDouble(1);
                double lon = rs.getDouble(2);
                return new LatLong(lat, lon);
            }
            rs.close();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        return null;
    }

    List<Long> findWayNodesByWayID(long id) {
        try {
            this.pStmtWayNodesR.setLong(1, id);
            ResultSet rs = this.pStmtWayNodesR.executeQuery();
            TreeMap<Integer, Long> nodeList = new TreeMap<Integer, Long>();
            while (rs.next()) {
                Long nodeID = rs.getLong(1);
                Integer pos = rs.getInt(2);
                nodeList.put(pos, nodeID);
            }
            rs.close();
            return new ArrayList<Long>(nodeList.values());
        }
        catch (SQLException e) {
            e.printStackTrace();
            return null;
        }
    }

    String getTagValue(Collection<Tag> tags, String key) {
        for (Tag tag : tags) {
            if (!tag.getKey().toLowerCase(Locale.ENGLISH).equals(key.toLowerCase(Locale.ENGLISH))) continue;
            return tag.getValue();
        }
        return null;
    }

    private String normalize(String str) {
        String normalized;
        if (ArabicNormalizer.isSpecialArabic(str) && (normalized = ArabicNormalizer.normalize(str)) != null && !normalized.trim().isEmpty()) {
            return normalized;
        }
        String normalizedString = Normalizer.normalize(str, Normalizer.Form.NFD);
        return NAME_NORMALIZE_PATTERN.matcher(normalizedString).replaceAll("");
    }

    private void postProcess() throws SQLException {
        LOGGER.info("Post-processing...");
        this.conn = DriverManager.getConnection("jdbc:sqlite:" + this.configuration.getOutputFile().getAbsolutePath());
        this.conn.createStatement().execute("DROP TABLE IF EXISTS nodes;");
        this.conn.createStatement().execute("DROP TABLE IF EXISTS waynodes;");
        this.conn.createStatement().execute("VACUUM;");
        this.conn.createStatement().execute("CREATE INDEX poi_index_idx_lat ON poi_index (lat);");
        this.conn.createStatement().execute("CREATE INDEX poi_index_idx_lon ON poi_index (lon);");
        this.conn.createStatement().execute("VACUUM;");
        this.conn.close();
    }

    private void prepareDatabase() throws ClassNotFoundException, SQLException, UnknownPoiCategoryException {
        Class.forName("org.sqlite.JDBC");
        this.conn = DriverManager.getConnection("jdbc:sqlite:" + this.configuration.getOutputFile().getAbsolutePath());
        this.conn.setAutoCommit(false);
        Statement stmt = this.conn.createStatement();
        stmt.execute("DROP TABLE IF EXISTS waynodes;");
        stmt.execute("DROP TABLE IF EXISTS nodes;");
        stmt.execute("DROP TABLE IF EXISTS metadata;");
        stmt.execute("DROP INDEX IF EXISTS poi_index_idx_lat;");
        stmt.execute("DROP INDEX IF EXISTS poi_index_idx_lon;");
        stmt.execute("DROP TABLE IF EXISTS poi_index;");
        stmt.execute("DROP TABLE IF EXISTS poi_category_map;");
        stmt.execute("DROP TABLE IF EXISTS poi_data;");
        stmt.execute("DROP TABLE IF EXISTS poi_categories;");
        stmt.execute("CREATE TABLE poi_categories (id INTEGER, name TEXT, parent INTEGER, PRIMARY KEY (id));");
        stmt.execute("CREATE TABLE poi_data (id INTEGER, data TEXT, PRIMARY KEY (id));");
        stmt.execute("CREATE TABLE poi_category_map (id INTEGER, category INTEGER, PRIMARY KEY (id, category)); ");
        stmt.execute("CREATE TABLE poi_index (id INTEGER, lat REAL, lon REAL, PRIMARY KEY (id));");
        stmt.execute("CREATE TABLE metadata (name TEXT, value TEXT);");
        stmt.execute("CREATE TABLE nodes (id INTEGER, lat REAL, lon REAL, PRIMARY KEY (id));");
        stmt.execute("CREATE TABLE waynodes (way INTEGER, node INTEGER, position INTEGER, PRIMARY KEY (way, node, position));");
        this.pStmtCatMap = this.conn.prepareStatement("INSERT INTO poi_category_map VALUES (?, ?);");
        this.pStmtData = this.conn.prepareStatement("INSERT INTO poi_data VALUES (?, ?);");
        this.pStmtIndex = this.conn.prepareStatement("INSERT INTO poi_index VALUES (?, ?, ?);");
        this.pStmtNodesC = this.conn.prepareStatement("INSERT INTO nodes VALUES (?, ?, ?);");
        this.pStmtNodesR = this.conn.prepareStatement("SELECT lat, lon FROM nodes WHERE id = ?;");
        this.pStmtWayNodesR = this.conn.prepareStatement("SELECT node, position FROM waynodes WHERE way = ?;");
        PreparedStatement pStmt = this.conn.prepareStatement("INSERT INTO poi_categories VALUES (?, ?, ?);");
        PoiCategory root = this.categoryManager.getRootCategory();
        pStmt.setLong(1, root.getID());
        pStmt.setString(2, root.getTitle());
        pStmt.setNull(3, 0);
        pStmt.addBatch();
        Stack<PoiCategory> children = new Stack<PoiCategory>();
        children.push(root);
        while (!children.isEmpty()) {
            for (PoiCategory c : ((PoiCategory)children.pop()).getChildren()) {
                pStmt.setLong(1, c.getID());
                pStmt.setString(2, c.getTitle());
                pStmt.setInt(3, c.getParent().getID());
                pStmt.addBatch();
                children.push(c);
            }
        }
        pStmt.executeBatch();
        this.conn.commit();
    }

    public void process(EntityContainer entityContainer) {
        Entity entity = entityContainer.getEntity();
        LOGGER.finest("Processing entity: " + entity.toString());
        switch (entity.getType()) {
            case Node: {
                Node node = (Node)entity;
                if (this.nNodes == 0L) {
                    LOGGER.info("Processing nodes...");
                }
                if (this.configuration.isProgressLogs() && this.nNodes % 100000L == 0L) {
                    System.out.print("Progress: Nodes " + this.nfCounts.format(this.nNodes) + " \r");
                }
                ++this.nNodes;
                if (this.configuration.isWays() && (!this.configuration.isWayFiltering() || node.getTags().isEmpty())) {
                    this.writeNode(node);
                }
                this.processEntity((Entity)node, node.getLatitude(), node.getLongitude());
                break;
            }
            case Way: {
                if (!this.configuration.isWays()) break;
                Way way = (Way)entity;
                if (this.nWays == 0L) {
                    LOGGER.info("Processing ways...");
                    try {
                        this.pStmtNodesC.executeBatch();
                        this.pStmtNodesC.clearBatch();
                    }
                    catch (SQLException e) {
                        e.printStackTrace();
                    }
                }
                if (this.configuration.isProgressLogs() && this.nWays % 10000L == 0L) {
                    System.out.print("Progress: Ways " + this.nfCounts.format(this.nWays) + " \r");
                }
                ++this.nWays;
                this.processWay(way);
                break;
            }
            case Relation: {
                if (!this.configuration.isGeoTags() || !this.configuration.isWays()) break;
                Relation relation = (Relation)entity;
                if (this.nRelations == 0L) {
                    LOGGER.info("Processing relations...");
                    this.geoTagger.commit();
                }
                this.geoTagger.filterBoundaries(relation);
                if (this.configuration.isProgressLogs() && this.nRelations % 10000L == 0L) {
                    System.out.print("Progress: Relations " + this.nfCounts.format(this.nRelations) + " \r");
                }
                ++this.nRelations;
                break;
            }
        }
        entity = null;
    }

    private void processEntity(Entity entity, double latitude, double longitude) {
        String name;
        BoundingBox bb = this.configuration.getBboxConfiguration();
        if (bb != null && !bb.contains(latitude, longitude)) {
            return;
        }
        if (entity.getTags().isEmpty()) {
            return;
        }
        if (this.configuration.isNames() && ((name = this.getTagValue(entity.getTags(), "name")) == null || name.isEmpty())) {
            return;
        }
        TreeMap<String, String> tagMap = new TreeMap<String, String>();
        HashSet<PoiCategory> categories = new HashSet<PoiCategory>();
        for (Tag tag : entity.getTags()) {
            String key = tag.getKey().toLowerCase(Locale.ENGLISH);
            if (!this.tagMappingResolver.getMappingTags().contains(key)) continue;
            String tagStr = key + "=" + tag.getValue();
            try {
                List<PoiCategory> pcs = this.tagMappingResolver.getCategoriesFromTag(tagStr);
                if (pcs == null) {
                    pcs = this.tagMappingResolver.getCategoriesFromTag(key);
                }
                if (pcs == null) continue;
                for (PoiCategory pc : pcs) {
                    if (!this.categoryFilter.isAcceptedCategory(pc)) continue;
                    if (tagMap.isEmpty()) {
                        for (Tag t : entity.getTags()) {
                            tagMap.put(t.getKey().toLowerCase(Locale.ENGLISH), t.getValue());
                            if (!this.configuration.isNormalize() || !t.getKey().toLowerCase(Locale.ENGLISH).equals("name")) continue;
                            String normalizedValue = this.normalize(t.getValue().toLowerCase(Locale.ROOT));
                            tagMap.put("normalized_name", normalizedValue);
                        }
                    }
                    categories.add(pc);
                }
            }
            catch (UnknownPoiCategoryException e) {
                LOGGER.warning("The '" + tagStr + "' tag refers to a POI that does not exist: " + e.toString());
            }
        }
        if (!tagMap.isEmpty()) {
            this.writePOI(this.poiAdded++, latitude, longitude, tagMap, categories);
        }
    }

    private void processWay(Way way) {
        LatLong centroid;
        if (way.getWayNodes().size() < 2) {
            LOGGER.finer("Found way with fewer than 2 way nodes. Way id: " + way.getId());
            return;
        }
        if (this.configuration.isGeoTags()) {
            this.geoTagger.storeAdministrativeBoundaries(way);
        }
        boolean validWay = true;
        LatLong[] wayNodes = new LatLong[way.getWayNodes().size()];
        int i = 0;
        for (WayNode wayNode : way.getWayNodes()) {
            wayNodes[i] = this.findNodeByID(wayNode.getNodeId());
            if (wayNodes[i] == null) {
                validWay = false;
                LOGGER.finer("Unknown way node " + wayNode.getNodeId() + " in way " + way.getId());
                break;
            }
            ++i;
        }
        if (!validWay) {
            return;
        }
        if (way.isClosed() && way.getWayNodes().size() >= 4) {
            Coordinate[] coordinates = new Coordinate[wayNodes.length];
            for (int j = 0; j < wayNodes.length; ++j) {
                LatLong wayNode = wayNodes[j];
                coordinates[j] = new Coordinate(wayNode.longitude, wayNode.latitude);
            }
            Polygon polygon = GEOMETRY_FACTORY.createPolygon(GEOMETRY_FACTORY.createLinearRing(coordinates), null);
            if (!polygon.isValid()) {
                return;
            }
            Point center = polygon.getCentroid();
            if (center == null) {
                return;
            }
            centroid = new LatLong(center.getY(), center.getX());
        } else {
            centroid = wayNodes[(wayNodes.length - 1) / 2];
        }
        this.processEntity((Entity)way, centroid.getLatitude(), centroid.getLongitude());
    }

    Map<String, String> stringToTags(String tagsmapstring) {
        String[] sb = tagsmapstring.split("\\r");
        HashMap<String, String> map = new HashMap<String, String>();
        for (String set : sb) {
            if (set.indexOf(61) <= -1) continue;
            String key = set.split(String.valueOf('='))[0];
            String value = set.substring(key.length() + 1);
            map.put(key, value);
        }
        return map;
    }

    String tagsToString(Map<String, String> tagMap) {
        StringBuilder sb = new StringBuilder();
        for (String key : tagMap.keySet()) {
            if (key.equalsIgnoreCase("created_by")) continue;
            if (sb.length() > 0) {
                sb.append('\r');
            }
            sb.append(key).append('=').append(tagMap.get(key));
        }
        return sb.toString();
    }

    private void writeMetadata() throws SQLException {
        LOGGER.info("Writing metadata...");
        PreparedStatement pStmtMetadata = this.conn.prepareStatement("INSERT INTO metadata VALUES (?, ?);");
        pStmtMetadata.setString(1, "bounds");
        BoundingBox bb = this.configuration.getBboxConfiguration();
        if (bb == null) {
            Statement stmt = this.conn.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT MIN(lat), MIN(lon), MAX(lat), MAX(lon) FROM poi_index;");
            rs.next();
            bb = new BoundingBox(rs.getDouble(1), rs.getDouble(2), rs.getDouble(3), rs.getDouble(4));
        }
        pStmtMetadata.setString(2, bb.minLatitude + "," + bb.minLongitude + "," + bb.maxLatitude + "," + bb.maxLongitude);
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "comment");
        if (this.configuration.getComment() != null) {
            pStmtMetadata.setString(2, this.configuration.getComment());
        } else {
            pStmtMetadata.setNull(2, 0);
        }
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "date");
        pStmtMetadata.setLong(2, System.currentTimeMillis());
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "language");
        if (!this.configuration.isAllTags() && this.configuration.getPreferredLanguage() != null) {
            pStmtMetadata.setString(2, this.configuration.getPreferredLanguage());
        } else {
            pStmtMetadata.setNull(2, 0);
        }
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "version");
        pStmtMetadata.setInt(2, this.configuration.getFileSpecificationVersion());
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "ways");
        pStmtMetadata.setString(2, Boolean.toString(this.configuration.isWays()));
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "writer");
        pStmtMetadata.setString(2, this.configuration.getWriterVersion());
        pStmtMetadata.addBatch();
        pStmtMetadata.executeBatch();
        this.conn.commit();
    }

    private void writeNode(Node node) {
        try {
            this.pStmtNodesC.setLong(1, node.getId());
            this.pStmtNodesC.setDouble(2, node.getLatitude());
            this.pStmtNodesC.setDouble(3, node.getLongitude());
            this.pStmtNodesC.addBatch();
            if (this.nNodes % 1024L == 0L) {
                this.pStmtNodesC.executeBatch();
                this.pStmtNodesC.clearBatch();
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    private void writePOI(long id, double latitude, double longitude, Map<String, String> poiData, Set<PoiCategory> categories) {
        try {
            this.pStmtIndex.setLong(1, id);
            this.pStmtIndex.setDouble(2, latitude);
            this.pStmtIndex.setDouble(3, longitude);
            this.pStmtIndex.addBatch();
            this.pStmtData.setLong(1, id);
            if (this.configuration.isAllTags()) {
                this.pStmtData.setString(2, this.tagsToString(poiData));
            } else {
                boolean foundPreferredLanguageName = false;
                String name = null;
                for (String key : poiData.keySet()) {
                    String language;
                    Matcher matcher;
                    if ("name".equals(key) && !foundPreferredLanguageName) {
                        name = poiData.get(key);
                        continue;
                    }
                    if (this.configuration.getPreferredLanguage() == null || foundPreferredLanguageName || !(matcher = NAME_LANGUAGE_PATTERN.matcher(key)).matches() || !(language = matcher.group(3)).equalsIgnoreCase(this.configuration.getPreferredLanguage())) continue;
                    name = poiData.get(key);
                    foundPreferredLanguageName = true;
                }
                if (name != null) {
                    this.pStmtData.setString(2, name);
                } else {
                    this.pStmtData.setNull(2, 0);
                }
            }
            this.pStmtData.addBatch();
            this.pStmtCatMap.setLong(1, id);
            for (PoiCategory category : categories) {
                this.pStmtCatMap.setInt(2, category.getID());
                this.pStmtCatMap.addBatch();
            }
            if (this.poiAdded % 1024L == 0L) {
                this.pStmtIndex.executeBatch();
                this.pStmtData.executeBatch();
                this.pStmtCatMap.executeBatch();
                this.pStmtIndex.clearBatch();
                this.pStmtData.clearBatch();
                this.pStmtCatMap.clearBatch();
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }
}

