/*
 * Decompiled with CFR 0.152.
 */
package org.mapsforge.poi.writer;

import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.Map;
import java.util.Stack;
import java.util.TreeMap;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.mapsforge.core.model.BoundingBox;
import org.mapsforge.core.model.LatLong;
import org.mapsforge.poi.storage.PoiCategory;
import org.mapsforge.poi.storage.PoiCategoryFilter;
import org.mapsforge.poi.storage.PoiCategoryManager;
import org.mapsforge.poi.storage.UnknownPoiCategoryException;
import org.mapsforge.poi.storage.WhitelistPoiCategoryFilter;
import org.mapsforge.poi.writer.TagMappingResolver;
import org.mapsforge.poi.writer.XMLPoiCategoryManager;
import org.mapsforge.poi.writer.logging.LoggerWrapper;
import org.mapsforge.poi.writer.logging.ProgressManager;
import org.mapsforge.poi.writer.model.PoiWriterConfiguration;
import org.openstreetmap.osmosis.core.container.v0_6.EntityContainer;
import org.openstreetmap.osmosis.core.domain.v0_6.Entity;
import org.openstreetmap.osmosis.core.domain.v0_6.Node;
import org.openstreetmap.osmosis.core.domain.v0_6.Tag;
import org.openstreetmap.osmosis.core.domain.v0_6.Way;
import org.openstreetmap.osmosis.core.domain.v0_6.WayNode;

public final class PoiWriter {
    private static final Logger LOGGER = LoggerWrapper.getLogger(PoiWriter.class.getName());
    private static final int BATCH_LIMIT = 1024;
    private static final int MIN_NODES_POLYGON = 4;
    private static final GeometryFactory GEOMETRY_FACTORY = new GeometryFactory();
    private static final Pattern NAME_LANGUAGE_PATTERN = Pattern.compile("(name)(:)([a-zA-Z]{1,3}(?:[-_][a-zA-Z0-9]{1,8})*)");
    private final PoiWriterConfiguration configuration;
    private final ProgressManager progressManager;
    private final PoiCategoryManager categoryManager;
    private final TagMappingResolver tagMappingResolver;
    private final PoiCategoryFilter categoryFilter;
    private int nNodes = 0;
    private int nWays = 0;
    private int poiAdded = 0;
    private Connection conn = null;
    private PreparedStatement pStmtData = null;
    private PreparedStatement pStmtIndex = null;
    private PreparedStatement pStmtNodesC = null;
    private PreparedStatement pStmtNodesR = null;

    public static PoiWriter newInstance(PoiWriterConfiguration configuration, ProgressManager progressManager) {
        return new PoiWriter(configuration, progressManager);
    }

    private PoiWriter(PoiWriterConfiguration configuration, ProgressManager progressManager) {
        this.configuration = configuration;
        this.progressManager = progressManager;
        LOGGER.info("Loading categories...");
        this.categoryManager = new XMLPoiCategoryManager(this.configuration.getTagMapping());
        this.tagMappingResolver = new TagMappingResolver(this.configuration.getTagMapping(), this.categoryManager);
        this.categoryFilter = new WhitelistPoiCategoryFilter();
        try {
            this.categoryFilter.addCategory(this.categoryManager.getRootCategory());
        }
        catch (UnknownPoiCategoryException e) {
            LOGGER.warning("Could not add category to filter: " + e.getMessage());
        }
        LOGGER.info("Adding tag mappings...");
        try {
            this.prepareDatabase();
        }
        catch (ClassNotFoundException | SQLException | UnknownPoiCategoryException e) {
            e.printStackTrace();
        }
        LOGGER.info("Creating POI database...");
        this.progressManager.initProgressBar(0, 0);
        this.progressManager.setMessage("Creating POI database");
    }

    private void commit() throws SQLException {
        LOGGER.info("Committing...");
        this.progressManager.setMessage("Committing...");
        this.pStmtIndex.executeBatch();
        this.pStmtData.executeBatch();
        this.conn.commit();
    }

    public void complete() {
        NumberFormat nfMegabyte = NumberFormat.getInstance();
        NumberFormat nfCounts = NumberFormat.getInstance();
        nfCounts.setGroupingUsed(true);
        nfMegabyte.setMaximumFractionDigits(2);
        try {
            this.commit();
            if (this.configuration.isFilterCategories()) {
                this.filterCategories();
            }
            this.writeMetadata();
            this.conn.close();
            this.postProcess();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        LOGGER.info("Added " + nfCounts.format(this.poiAdded) + " POIs.");
        this.progressManager.setMessage("Done.");
        LOGGER.info("Estimated memory consumption: " + nfMegabyte.format((double)(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) / Math.pow(1024.0, 2.0)) + "MB");
    }

    private void filterCategories() throws SQLException {
        LOGGER.info("Filtering categories...");
        PreparedStatement pStmtChildren = this.conn.prepareStatement("SELECT COUNT(*) FROM poi_categories WHERE parent = ?;");
        PreparedStatement pStmtPoi = this.conn.prepareStatement("SELECT COUNT(*) FROM poi_data WHERE category = ?;");
        PreparedStatement pStmtDel = this.conn.prepareStatement("DELETE FROM poi_categories WHERE id = ?;");
        Statement stmt = this.conn.createStatement();
        ResultSet rs = stmt.executeQuery("SELECT id FROM poi_categories ORDER BY id;");
        while (rs.next()) {
            long nPoi;
            long nChildren;
            int id = rs.getInt(1);
            pStmtChildren.setInt(1, id);
            ResultSet rsChildren = pStmtChildren.executeQuery();
            if (!rsChildren.next() || (nChildren = rsChildren.getLong(1)) != 0L) continue;
            pStmtPoi.setInt(1, id);
            ResultSet rsPoi = pStmtPoi.executeQuery();
            if (!rsPoi.next() || (nPoi = rsPoi.getLong(1)) != 0L) continue;
            pStmtDel.setInt(1, id);
            pStmtDel.executeUpdate();
        }
    }

    public LatLong findNodeByID(long id) {
        try {
            this.pStmtNodesR.setLong(1, id);
            ResultSet rs = this.pStmtNodesR.executeQuery();
            if (rs.next()) {
                double lat = rs.getDouble(1);
                double lon = rs.getDouble(2);
                return new LatLong(lat, lon);
            }
            rs.close();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        return null;
    }

    private void postProcess() throws SQLException {
        LOGGER.info("Post-processing...");
        this.conn = DriverManager.getConnection("jdbc:sqlite:" + this.configuration.getOutputFile().getAbsolutePath());
        this.conn.createStatement().execute("DROP TABLE IF EXISTS nodes;");
        this.conn.close();
        this.conn = DriverManager.getConnection("jdbc:sqlite:" + this.configuration.getOutputFile().getAbsolutePath());
        this.conn.createStatement().execute("VACUUM;");
        this.conn.close();
    }

    private void prepareDatabase() throws ClassNotFoundException, SQLException, UnknownPoiCategoryException {
        Class.forName("org.sqlite.JDBC");
        this.conn = DriverManager.getConnection("jdbc:sqlite:" + this.configuration.getOutputFile().getAbsolutePath());
        this.conn.setAutoCommit(false);
        Statement stmt = this.conn.createStatement();
        stmt.execute("DROP TABLE IF EXISTS nodes;");
        stmt.execute("DROP TABLE IF EXISTS metadata;");
        stmt.execute("DROP TABLE IF EXISTS poi_index;");
        stmt.execute("DROP TABLE IF EXISTS poi_data;");
        stmt.execute("DROP TABLE IF EXISTS poi_categories;");
        stmt.execute("CREATE TABLE poi_categories (id INTEGER, name TEXT, parent INTEGER, PRIMARY KEY (id));");
        stmt.execute("CREATE TABLE poi_data (id INTEGER, data TEXT, category INTEGER, PRIMARY KEY (id));");
        stmt.execute("CREATE VIRTUAL TABLE poi_index USING rtree(id, minLat, maxLat, minLon, maxLon);");
        stmt.execute("CREATE TABLE metadata (name TEXT, value TEXT);");
        stmt.execute("CREATE TABLE nodes (id INTEGER, lat REAL, lon REAL, PRIMARY KEY (id));");
        this.pStmtData = this.conn.prepareStatement("INSERT INTO poi_data VALUES (?, ?, ?);");
        this.pStmtIndex = this.conn.prepareStatement("INSERT INTO poi_index VALUES (?, ?, ?, ?, ?);");
        this.pStmtNodesC = this.conn.prepareStatement("INSERT INTO nodes VALUES (?, ?, ?);");
        this.pStmtNodesR = this.conn.prepareStatement("SELECT lat, lon FROM nodes WHERE id = ?;");
        PreparedStatement pStmt = this.conn.prepareStatement("INSERT INTO poi_categories VALUES (?, ?, ?);");
        PoiCategory root = this.categoryManager.getRootCategory();
        pStmt.setLong(1, root.getID());
        pStmt.setString(2, root.getTitle());
        pStmt.setNull(3, 0);
        pStmt.addBatch();
        Stack<PoiCategory> children = new Stack<PoiCategory>();
        children.push(root);
        while (!children.isEmpty()) {
            for (PoiCategory c : ((PoiCategory)children.pop()).getChildren()) {
                pStmt.setLong(1, c.getID());
                pStmt.setString(2, c.getTitle());
                pStmt.setInt(3, c.getParent().getID());
                pStmt.addBatch();
                children.push(c);
            }
        }
        pStmt.executeBatch();
        this.conn.commit();
    }

    public void process(EntityContainer entityContainer) {
        Entity entity = entityContainer.getEntity();
        LOGGER.finest("Processing entity: " + entity.toString());
        switch (entity.getType()) {
            case Node: {
                Node node = (Node)entity;
                if (this.nNodes == 0) {
                    LOGGER.info("Processing nodes...");
                }
                ++this.nNodes;
                if (this.configuration.isWays()) {
                    this.writeNode(node);
                }
                this.processEntity((Entity)node, node.getLatitude(), node.getLongitude());
                break;
            }
            case Way: {
                if (!this.configuration.isWays()) break;
                Way way = (Way)entity;
                if (this.nWays == 0) {
                    LOGGER.info("Processing ways...");
                    try {
                        this.pStmtNodesC.executeBatch();
                        this.pStmtNodesC.clearBatch();
                    }
                    catch (SQLException e) {
                        e.printStackTrace();
                    }
                }
                ++this.nWays;
                this.processWay(way);
            }
        }
        entity = null;
    }

    private void processEntity(Entity entity, double latitude, double longitude) {
        BoundingBox bb = this.configuration.getBboxConfiguration();
        if (bb != null && !bb.contains(latitude, longitude)) {
            return;
        }
        if (entity.getTags().isEmpty()) {
            return;
        }
        TreeMap<String, String> tagMap = new TreeMap<String, String>();
        String tagStr = null;
        for (Tag tag : entity.getTags()) {
            String key = tag.getKey().toLowerCase(Locale.ENGLISH);
            if (this.tagMappingResolver.getMappingTags().contains(key)) {
                tagStr = key + "=" + tag.getValue();
            }
            tagMap.put(key, tag.getValue());
        }
        try {
            PoiCategory pc = null;
            if (tagStr != null) {
                pc = this.tagMappingResolver.getCategoryFromTag(tagStr);
            }
            if (pc != null && this.categoryFilter.isAcceptedCategory(pc)) {
                ++this.poiAdded;
                this.writePOI(this.poiAdded, latitude, longitude, tagMap, pc);
            }
        }
        catch (UnknownPoiCategoryException e) {
            LOGGER.warning("The '" + tagStr + "' tag refers to a POI that does not exist: " + e.getMessage());
        }
    }

    private void processWay(Way way) {
        if (!way.isClosed()) {
            return;
        }
        if (way.getWayNodes().size() < 4) {
            LOGGER.finer("Found closed polygon with fewer than 4 way nodes. Way id: " + way.getId());
            return;
        }
        boolean validWay = true;
        LatLong[] wayNodes = new LatLong[way.getWayNodes().size()];
        int i = 0;
        for (WayNode wayNode : way.getWayNodes()) {
            wayNodes[i] = this.findNodeByID(wayNode.getNodeId());
            if (wayNodes[i] == null) {
                validWay = false;
                LOGGER.finer("Unknown way node " + wayNode.getNodeId() + " in way " + way.getId());
                break;
            }
            ++i;
        }
        if (!validWay) {
            return;
        }
        Coordinate[] coordinates = new Coordinate[wayNodes.length];
        for (int j = 0; j < wayNodes.length; ++j) {
            LatLong wayNode = wayNodes[j];
            coordinates[j] = new Coordinate(wayNode.longitude, wayNode.latitude);
        }
        Polygon polygon = GEOMETRY_FACTORY.createPolygon(GEOMETRY_FACTORY.createLinearRing(coordinates), null);
        Point centroid = polygon.getCentroid();
        if (centroid == null) {
            return;
        }
        this.processEntity((Entity)way, centroid.getY(), centroid.getX());
    }

    private String tagsToString(Map<String, String> tagMap) {
        StringBuilder sb = new StringBuilder();
        for (String key : tagMap.keySet()) {
            if (key.equalsIgnoreCase("created_by")) continue;
            if (sb.length() > 0) {
                sb.append('\r');
            }
            sb.append(key).append('=').append(tagMap.get(key));
        }
        return sb.toString();
    }

    private void writeMetadata() throws SQLException {
        LOGGER.info("Writing metadata...");
        PreparedStatement pStmtMetadata = this.conn.prepareStatement("INSERT INTO metadata VALUES (?, ?);");
        pStmtMetadata.setString(1, "bounds");
        BoundingBox bb = this.configuration.getBboxConfiguration();
        if (bb != null) {
            pStmtMetadata.setString(2, bb.minLatitude + "," + bb.minLongitude + "," + bb.maxLatitude + "," + bb.maxLongitude);
        } else {
            pStmtMetadata.setNull(2, 0);
        }
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "comment");
        if (this.configuration.getComment() != null) {
            pStmtMetadata.setString(2, this.configuration.getComment());
        } else {
            pStmtMetadata.setNull(2, 0);
        }
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "date");
        pStmtMetadata.setLong(2, System.currentTimeMillis());
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "language");
        if (!this.configuration.isAllTags() && this.configuration.getPreferredLanguage() != null) {
            pStmtMetadata.setString(2, this.configuration.getPreferredLanguage());
        } else {
            pStmtMetadata.setNull(2, 0);
        }
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "version");
        pStmtMetadata.setInt(2, this.configuration.getFileSpecificationVersion());
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "ways");
        pStmtMetadata.setString(2, Boolean.toString(this.configuration.isWays()));
        pStmtMetadata.addBatch();
        pStmtMetadata.setString(1, "writer");
        pStmtMetadata.setString(2, this.configuration.getWriterVersion());
        pStmtMetadata.addBatch();
        pStmtMetadata.executeBatch();
        this.conn.commit();
    }

    private void writeNode(Node node) {
        try {
            this.pStmtNodesC.setLong(1, node.getId());
            this.pStmtNodesC.setDouble(2, node.getLatitude());
            this.pStmtNodesC.setDouble(3, node.getLongitude());
            this.pStmtNodesC.addBatch();
            if (this.nNodes % 1024 == 0) {
                this.pStmtNodesC.executeBatch();
                this.pStmtNodesC.clearBatch();
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    private void writePOI(long id, double latitude, double longitude, Map<String, String> poiData, PoiCategory category) {
        try {
            this.pStmtIndex.setLong(1, id);
            this.pStmtIndex.setDouble(2, latitude);
            this.pStmtIndex.setDouble(3, latitude);
            this.pStmtIndex.setDouble(4, longitude);
            this.pStmtIndex.setDouble(5, longitude);
            this.pStmtData.setLong(1, id);
            if (this.configuration.isAllTags()) {
                this.pStmtData.setString(2, this.tagsToString(poiData));
            } else {
                boolean foundPreferredLanguageName = false;
                String name = null;
                for (String key : poiData.keySet()) {
                    String language;
                    Matcher matcher;
                    if ("name".equals(key) && !foundPreferredLanguageName) {
                        name = poiData.get(key);
                        continue;
                    }
                    if (this.configuration.getPreferredLanguage() == null || foundPreferredLanguageName || !(matcher = NAME_LANGUAGE_PATTERN.matcher(key)).matches() || !(language = matcher.group(3)).equalsIgnoreCase(this.configuration.getPreferredLanguage())) continue;
                    name = poiData.get(key);
                    foundPreferredLanguageName = true;
                }
                if (name != null) {
                    this.pStmtData.setString(2, name);
                } else {
                    this.pStmtData.setNull(2, 0);
                }
            }
            this.pStmtData.setInt(3, category.getID());
            this.pStmtIndex.addBatch();
            this.pStmtData.addBatch();
            if (this.poiAdded % 1024 == 0) {
                this.pStmtIndex.executeBatch();
                this.pStmtData.executeBatch();
                this.pStmtIndex.clearBatch();
                this.pStmtData.clearBatch();
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }
}

