/*
 * Decompiled with CFR 0.152.
 */
package org.oscim.tiling.source.mapfile;

import java.io.IOException;
import java.io.RandomAccessFile;
import org.oscim.backend.CanvasAdapter;
import org.oscim.core.GeometryBuffer;
import org.oscim.core.MapElement;
import org.oscim.core.MercatorProjection;
import org.oscim.core.Tag;
import org.oscim.core.Tile;
import org.oscim.layers.tile.MapTile;
import org.oscim.tiling.ITileDataSink;
import org.oscim.tiling.ITileDataSource;
import org.oscim.tiling.QueryResult;
import org.oscim.tiling.source.mapfile.MapFileTileSource;
import org.oscim.tiling.source.mapfile.OSMUtils;
import org.oscim.tiling.source.mapfile.Projection;
import org.oscim.tiling.source.mapfile.QueryCalculations;
import org.oscim.tiling.source.mapfile.QueryParameters;
import org.oscim.tiling.source.mapfile.ReadBuffer;
import org.oscim.tiling.source.mapfile.header.SubFileParameter;
import org.oscim.utils.geom.TileClipper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MapDatabase
implements ITileDataSource {
    private static final long BITMASK_INDEX_OFFSET = 0x7FFFFFFFFFL;
    private static final long BITMASK_INDEX_WATER = 0x8000000000L;
    private static final String DEBUG_SIGNATURE_BLOCK = "block signature: ";
    private static final String DEBUG_SIGNATURE_WAY = "way signature: ";
    private static final String INVALID_FIRST_WAY_OFFSET = "invalid first way offset: ";
    static final Logger log = LoggerFactory.getLogger(MapDatabase.class);
    private static final int POI_FEATURE_ELEVATION = 32;
    private static final int POI_FEATURE_HOUSE_NUMBER = 64;
    private static final int POI_FEATURE_NAME = 128;
    private static final int POI_LAYER_BITMASK = 240;
    private static final int POI_LAYER_SHIFT = 4;
    private static final int POI_NUMBER_OF_TAGS_BITMASK = 15;
    private static final byte SIGNATURE_LENGTH_BLOCK = 32;
    private static final byte SIGNATURE_LENGTH_POI = 32;
    private static final byte SIGNATURE_LENGTH_WAY = 32;
    private static final int WAY_FEATURE_DATA_BLOCKS_BYTE = 8;
    private static final int WAY_FEATURE_DOUBLE_DELTA_ENCODING = 4;
    private static final int WAY_FEATURE_HOUSE_NUMBER = 64;
    private static final int WAY_FEATURE_LABEL_POSITION = 16;
    private static final int WAY_FEATURE_NAME = 128;
    private static final int WAY_FEATURE_REF = 32;
    private static final int WAY_LAYER_BITMASK = 240;
    private static final int WAY_LAYER_SHIFT = 4;
    private static final int WAY_NUMBER_OF_TAGS_BITMASK = 15;
    private long mFileSize;
    private boolean mDebugFile;
    private RandomAccessFile mInputFile;
    private ReadBuffer mReadBuffer;
    private String mSignatureBlock;
    private String mSignaturePoi;
    private String mSignatureWay;
    private int mTileLatitude;
    private int mTileLongitude;
    private int[] mIntBuffer;
    private final MapElement mElem = new MapElement();
    private int minDeltaLat;
    private int minDeltaLon;
    private final TileProjection mTileProjection;
    private final TileClipper mTileClipper;
    private final MapFileTileSource mTileSource;
    private int xmin;
    private int ymin;
    private int xmax;
    private int ymax;
    private int stringOffset = -1;

    public MapDatabase(MapFileTileSource tileSource) throws IOException {
        this.mTileSource = tileSource;
        try {
            this.mInputFile = new RandomAccessFile(tileSource.mapFile, "r");
            this.mFileSize = this.mInputFile.length();
            this.mReadBuffer = new ReadBuffer(this.mInputFile);
        }
        catch (IOException e) {
            log.error(e.getMessage());
            this.dispose();
            throw new IOException();
        }
        this.mTileProjection = new TileProjection();
        this.mTileClipper = new TileClipper(0.0f, 0.0f, 0.0f, 0.0f);
    }

    public MapFileTileSource getTileSource() {
        return this.mTileSource;
    }

    @Override
    public void query(MapTile tile, ITileDataSink sink) {
        if (this.mTileSource.fileHeader == null) {
            sink.completed(QueryResult.FAILED);
            return;
        }
        if (this.mIntBuffer == null) {
            this.mIntBuffer = new int[65534];
        }
        try {
            this.mTileProjection.setTile(tile);
            double size = 1.0 / (double)(1 << tile.zoomLevel);
            int pixel = tile.zoomLevel > 11 ? 1 : 2;
            int simplify = Tile.SIZE / pixel;
            this.minDeltaLat = (int)(Math.abs(MercatorProjection.toLatitude(tile.y + size) - MercatorProjection.toLatitude(tile.y)) * 1000000.0) / simplify;
            this.minDeltaLon = (int)(Math.abs(MercatorProjection.toLongitude(tile.x + size) - MercatorProjection.toLongitude(tile.x)) * 1000000.0) / simplify;
            QueryParameters queryParameters = new QueryParameters();
            queryParameters.queryZoomLevel = this.mTileSource.fileHeader.getQueryZoomLevel(tile.zoomLevel);
            SubFileParameter subFileParameter = this.mTileSource.fileHeader.getSubFileParameter(queryParameters.queryZoomLevel);
            if (subFileParameter == null) {
                log.warn("no sub-file for zoom level: " + queryParameters.queryZoomLevel);
                sink.completed(QueryResult.FAILED);
                return;
            }
            QueryCalculations.calculateBaseTiles(queryParameters, tile, subFileParameter);
            QueryCalculations.calculateBlocks(queryParameters, subFileParameter);
            this.processBlocks(sink, queryParameters, subFileParameter);
        }
        catch (IOException e) {
            log.error(e.getMessage());
            sink.completed(QueryResult.FAILED);
            return;
        }
        sink.completed(QueryResult.SUCCESS);
    }

    @Override
    public void dispose() {
        this.mReadBuffer = null;
        if (this.mInputFile != null) {
            try {
                this.mInputFile.close();
                this.mInputFile = null;
            }
            catch (IOException e) {
                log.error(e.getMessage());
            }
        }
    }

    @Override
    public void cancel() {
    }

    private void logDebugSignatures() {
        if (this.mDebugFile) {
            log.warn(DEBUG_SIGNATURE_WAY + this.mSignatureWay);
            log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
        }
    }

    private void processBlock(QueryParameters queryParameters, SubFileParameter subFileParameter, ITileDataSink mapDataSink) {
        if (!this.processBlockSignature()) {
            return;
        }
        int[][] zoomTable = this.readZoomTable(subFileParameter);
        if (zoomTable == null) {
            return;
        }
        int zoomTableRow = queryParameters.queryZoomLevel - subFileParameter.zoomLevelMin;
        int poisOnQueryZoomLevel = zoomTable[zoomTableRow][0];
        int waysOnQueryZoomLevel = zoomTable[zoomTableRow][1];
        int firstWayOffset = this.mReadBuffer.readUnsignedInt();
        if (firstWayOffset < 0) {
            log.warn(INVALID_FIRST_WAY_OFFSET + firstWayOffset);
            if (this.mDebugFile) {
                log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
            }
            return;
        }
        if ((firstWayOffset += this.mReadBuffer.getBufferPosition()) > this.mReadBuffer.getBufferSize()) {
            log.warn(INVALID_FIRST_WAY_OFFSET + firstWayOffset);
            if (this.mDebugFile) {
                log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
            }
            return;
        }
        if (!this.processPOIs(mapDataSink, poisOnQueryZoomLevel)) {
            return;
        }
        if (this.mReadBuffer.getBufferPosition() > firstWayOffset) {
            log.warn("invalid buffer position: " + this.mReadBuffer.getBufferPosition());
            if (this.mDebugFile) {
                log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
            }
            return;
        }
        this.mReadBuffer.setBufferPosition(firstWayOffset);
        if (!this.processWays(queryParameters, mapDataSink, waysOnQueryZoomLevel)) {
            return;
        }
    }

    private void setTileClipping(QueryParameters queryParameters, long mCurrentRow, long mCurrentCol) {
        long numRows = queryParameters.toBlockY - queryParameters.fromBlockY;
        long numCols = queryParameters.toBlockX - queryParameters.fromBlockX;
        int buffer = (int)(16.0f * CanvasAdapter.dpi / 160.0f + 0.5f);
        this.xmin = -buffer;
        this.ymin = -buffer;
        this.xmax = Tile.SIZE + buffer;
        this.ymax = Tile.SIZE + buffer;
        if (numRows > 0L) {
            int w = (int)((long)Tile.SIZE / (numCols + 1L));
            int h = (int)((long)Tile.SIZE / (numRows + 1L));
            if (mCurrentCol > 0L) {
                this.xmin = (int)(mCurrentCol * (long)w);
            }
            if (mCurrentCol < numCols) {
                this.xmax = (int)(mCurrentCol * (long)w + (long)w);
            }
            if (mCurrentRow > 0L) {
                this.ymin = (int)(mCurrentRow * (long)h);
            }
            if (mCurrentRow < numRows) {
                this.ymax = (int)(mCurrentRow * (long)h + (long)h);
            }
        }
        this.mTileClipper.setRect(this.xmin, this.ymin, this.xmax, this.ymax);
    }

    private void processBlocks(ITileDataSink mapDataSink, QueryParameters queryParams, SubFileParameter subFileParameter) throws IOException {
        for (long row = queryParams.fromBlockY; row <= queryParams.toBlockY; ++row) {
            for (long column = queryParams.fromBlockX; column <= queryParams.toBlockX; ++column) {
                long nextBlockPointer;
                long blockPointer;
                this.setTileClipping(queryParams, row - queryParams.fromBlockY, column - queryParams.fromBlockX);
                long blockNumber = row * subFileParameter.blocksWidth + column;
                long blockIndexEntry = this.mTileSource.databaseIndexCache.getIndexEntry(subFileParameter, blockNumber);
                if ((blockIndexEntry & 0x8000000000L) != 0L) {
                    // empty if block
                }
                if ((blockPointer = blockIndexEntry & 0x7FFFFFFFFFL) < 1L || blockPointer > subFileParameter.subFileSize) {
                    log.warn("invalid current block pointer: " + blockPointer);
                    log.warn("subFileSize: " + subFileParameter.subFileSize);
                    return;
                }
                if (blockNumber + 1L == subFileParameter.numberOfBlocks) {
                    nextBlockPointer = subFileParameter.subFileSize;
                } else {
                    nextBlockPointer = this.mTileSource.databaseIndexCache.getIndexEntry(subFileParameter, blockNumber + 1L);
                    if ((nextBlockPointer &= 0x7FFFFFFFFFL) < 1L || nextBlockPointer > subFileParameter.subFileSize) {
                        log.warn("invalid next block pointer: " + nextBlockPointer);
                        log.warn("sub-file size: " + subFileParameter.subFileSize);
                        return;
                    }
                }
                int blockSize = (int)(nextBlockPointer - blockPointer);
                if (blockSize < 0) {
                    log.warn("current block size must not be negative: " + blockSize);
                    return;
                }
                if (blockSize == 0) continue;
                if (blockSize > 8000000) {
                    log.warn("current block size too large: " + blockSize);
                    continue;
                }
                if (blockPointer + (long)blockSize > this.mFileSize) {
                    log.warn("current block larger than file size: " + blockSize);
                    return;
                }
                this.mInputFile.seek(subFileParameter.startAddress + blockPointer);
                if (!this.mReadBuffer.readFromFile(blockSize)) {
                    log.warn("reading current block has failed: " + blockSize);
                    return;
                }
                double tileLatitudeDeg = Projection.tileYToLatitude(subFileParameter.boundaryTileTop + row, subFileParameter.baseZoomLevel);
                double tileLongitudeDeg = Projection.tileXToLongitude(subFileParameter.boundaryTileLeft + column, subFileParameter.baseZoomLevel);
                this.mTileLatitude = (int)(tileLatitudeDeg * 1000000.0);
                this.mTileLongitude = (int)(tileLongitudeDeg * 1000000.0);
                this.processBlock(queryParams, subFileParameter, mapDataSink);
            }
        }
    }

    private boolean processBlockSignature() {
        if (this.mDebugFile) {
            this.mSignatureBlock = this.mReadBuffer.readUTF8EncodedString(32);
            if (!this.mSignatureBlock.startsWith("###TileStart")) {
                log.warn("invalid block signature: " + this.mSignatureBlock);
                return false;
            }
        }
        return true;
    }

    private boolean processPOIs(ITileDataSink mapDataSink, int numberOfPois) {
        Tag[] poiTags = this.mTileSource.fileInfo.poiTags;
        MapElement e = this.mElem;
        for (int elementCounter = numberOfPois; elementCounter != 0; --elementCounter) {
            String str;
            int numTags = 0;
            if (this.mDebugFile) {
                this.mSignaturePoi = this.mReadBuffer.readUTF8EncodedString(32);
                if (!this.mSignaturePoi.startsWith("***POIStart")) {
                    log.warn("invalid POI signature: " + this.mSignaturePoi);
                    log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
                    return false;
                }
            }
            int latitude = this.mTileLatitude + this.mReadBuffer.readSignedInt();
            int longitude = this.mTileLongitude + this.mReadBuffer.readSignedInt();
            byte specialByte = this.mReadBuffer.readByte();
            byte layer = (byte)((specialByte & 0xF0) >>> 4);
            byte numberOfTags = (byte)(specialByte & 0xF);
            if (numberOfTags != 0) {
                if (!this.mReadBuffer.readTags(e.tags, poiTags, numberOfTags)) {
                    return false;
                }
                numTags = numberOfTags;
            }
            e.tags.numTags = numTags;
            byte featureByte = this.mReadBuffer.readByte();
            if ((featureByte & 0x80) != 0) {
                str = this.mTileSource.extractLocalized(this.mReadBuffer.readUTF8EncodedString());
                e.tags.add(new Tag("name", str, false));
            }
            if ((featureByte & 0x40) != 0) {
                str = this.mReadBuffer.readUTF8EncodedString();
                e.tags.add(new Tag("addr:housenumber", str, false));
            }
            if ((featureByte & 0x20) != 0) {
                str = Integer.toString(this.mReadBuffer.readSignedInt());
                e.tags.add(new Tag("ele", str, false));
            }
            this.mTileProjection.projectPoint(latitude, longitude, e);
            e.setLayer(layer);
            mapDataSink.process(e);
        }
        return true;
    }

    private boolean processWayDataBlock(MapElement e, boolean doubleDeltaEncoding, boolean isLine) {
        int numBlocks = this.mReadBuffer.readUnsignedInt();
        if (numBlocks < 1 || numBlocks > Short.MAX_VALUE) {
            log.warn("invalid number of way coordinate blocks: " + numBlocks);
            return false;
        }
        int[] wayLengths = e.ensureIndexSize(numBlocks, false);
        if (wayLengths.length > numBlocks) {
            wayLengths[numBlocks] = -1;
        }
        for (int coordinateBlock = 0; coordinateBlock < numBlocks; ++coordinateBlock) {
            int numWayNodes = this.mReadBuffer.readUnsignedInt();
            if (numWayNodes < 2 || numWayNodes > Short.MAX_VALUE) {
                log.warn("invalid number of way nodes: " + numWayNodes);
                this.logDebugSignatures();
                return false;
            }
            int len = numWayNodes * 2;
            wayLengths[coordinateBlock] = this.decodeWayNodes(doubleDeltaEncoding, e, len, isLine);
        }
        return true;
    }

    private int decodeWayNodes(boolean doubleDelta, MapElement e, int length, boolean isLine) {
        int lon;
        int lat;
        int[] buffer = this.mIntBuffer;
        this.mReadBuffer.readSignedInt(buffer, length);
        float[] outBuffer = e.ensurePointSize(e.pointPos + length, true);
        int outPos = e.pointPos;
        int firstLat = lat = this.mTileLatitude + buffer[0];
        int firstLon = lon = this.mTileLongitude + buffer[1];
        outBuffer[outPos++] = lon;
        outBuffer[outPos++] = lat;
        int cnt = 2;
        int deltaLat = 0;
        int deltaLon = 0;
        for (int pos = 2; pos < length; pos += 2) {
            if (doubleDelta) {
                deltaLat = buffer[pos] + deltaLat;
                deltaLon = buffer[pos + 1] + deltaLon;
            } else {
                deltaLat = buffer[pos];
                deltaLon = buffer[pos + 1];
            }
            lat += deltaLat;
            lon += deltaLon;
            if (pos == length - 2) {
                boolean line;
                boolean bl = line = isLine || lon != firstLon && lat != firstLat;
                if (line) {
                    outBuffer[outPos++] = lon;
                    outBuffer[outPos++] = lat;
                    cnt += 2;
                }
                if (e.type != GeometryBuffer.GeometryType.NONE) continue;
                e.type = line ? GeometryBuffer.GeometryType.LINE : GeometryBuffer.GeometryType.POLY;
                continue;
            }
            outBuffer[outPos++] = lon;
            outBuffer[outPos++] = lat;
            cnt += 2;
        }
        e.pointPos = outPos;
        return cnt;
    }

    private boolean processWays(QueryParameters queryParameters, ITileDataSink mapDataSink, int numberOfWays) {
        Tag[] wayTags = this.mTileSource.fileInfo.wayTags;
        MapElement e = this.mElem;
        int stringsSize = 0;
        this.stringOffset = 0;
        if (this.mTileSource.experimental) {
            stringsSize = this.mReadBuffer.readUnsignedInt();
            this.stringOffset = this.mReadBuffer.getBufferPosition();
            this.mReadBuffer.skipBytes(stringsSize);
        }
        for (int elementCounter = numberOfWays; elementCounter != 0; --elementCounter) {
            int wayDataBlocks;
            byte featureByte;
            byte numTags = 0;
            if (this.mDebugFile) {
                this.mSignatureWay = this.mReadBuffer.readUTF8EncodedString(32);
                if (!this.mSignatureWay.startsWith("---WayStart")) {
                    log.warn("invalid way signature: " + this.mSignatureWay);
                    log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
                    return false;
                }
            }
            if (queryParameters.useTileBitmask) {
                if ((elementCounter = this.mReadBuffer.skipWays(queryParameters.queryTileBitmask, elementCounter)) == 0) {
                    return true;
                }
                if (elementCounter < 0) {
                    return false;
                }
                if (this.mTileSource.experimental && this.mReadBuffer.lastTagPosition > 0) {
                    int pos = this.mReadBuffer.getBufferPosition();
                    this.mReadBuffer.setBufferPosition(this.mReadBuffer.lastTagPosition);
                    byte numberOfTags = (byte)(this.mReadBuffer.readByte() & 0xF);
                    if (!this.mReadBuffer.readTags(e.tags, wayTags, numberOfTags)) {
                        return false;
                    }
                    numTags = numberOfTags;
                    this.mReadBuffer.setBufferPosition(pos);
                }
            } else {
                int wayDataSize = this.mReadBuffer.readUnsignedInt();
                if (wayDataSize < 0) {
                    log.warn("invalid way data size: " + wayDataSize);
                    if (this.mDebugFile) {
                        log.warn(DEBUG_SIGNATURE_BLOCK + this.mSignatureBlock);
                    }
                    log.error("BUG way 2");
                    return false;
                }
                this.mReadBuffer.skipBytes(2);
            }
            byte specialByte = this.mReadBuffer.readByte();
            byte layer = (byte)((specialByte & 0xF0) >>> 4);
            byte numberOfTags = (byte)(specialByte & 0xF);
            if (numberOfTags != 0) {
                if (!this.mReadBuffer.readTags(e.tags, wayTags, numberOfTags)) {
                    return false;
                }
                numTags = numberOfTags;
            }
            boolean featureWayDoubleDeltaEncoding = ((featureByte = this.mReadBuffer.readByte()) & 4) != 0;
            boolean hasName = (featureByte & 0x80) != 0;
            boolean hasHouseNr = (featureByte & 0x40) != 0;
            boolean hasRef = (featureByte & 0x20) != 0;
            e.tags.numTags = numTags;
            if (this.mTileSource.experimental) {
                String str;
                int textPos;
                if (hasName) {
                    textPos = this.mReadBuffer.readUnsignedInt();
                    str = this.mTileSource.extractLocalized(this.mReadBuffer.readUTF8EncodedStringAt(this.stringOffset + textPos));
                    e.tags.add(new Tag("name", str, false));
                }
                if (hasHouseNr) {
                    textPos = this.mReadBuffer.readUnsignedInt();
                    str = this.mReadBuffer.readUTF8EncodedStringAt(this.stringOffset + textPos);
                    e.tags.add(new Tag("addr:housenumber", str, false));
                }
                if (hasRef) {
                    textPos = this.mReadBuffer.readUnsignedInt();
                    str = this.mReadBuffer.readUTF8EncodedStringAt(this.stringOffset + textPos);
                    e.tags.add(new Tag("ref", str, false));
                }
            } else {
                String str;
                if (hasName) {
                    str = this.mTileSource.extractLocalized(this.mReadBuffer.readUTF8EncodedString());
                    e.tags.add(new Tag("name", str, false));
                }
                if (hasHouseNr) {
                    str = this.mReadBuffer.readUTF8EncodedString();
                    e.tags.add(new Tag("addr:housenumber", str, false));
                }
                if (hasRef) {
                    str = this.mReadBuffer.readUTF8EncodedString();
                    e.tags.add(new Tag("ref", str, false));
                }
            }
            int[] labelPosition = null;
            if ((featureByte & 0x10) != 0) {
                labelPosition = this.readOptionalLabelPosition();
            }
            if ((featureByte & 8) != 0) {
                wayDataBlocks = this.mReadBuffer.readUnsignedInt();
                if (wayDataBlocks < 1) {
                    log.warn("invalid number of way data blocks: " + wayDataBlocks);
                    this.logDebugSignatures();
                    return false;
                }
            } else {
                wayDataBlocks = 1;
            }
            boolean linearFeature = !OSMUtils.isArea(e);
            for (int wayDataBlock = 0; wayDataBlock < wayDataBlocks; ++wayDataBlock) {
                e.clear();
                if (!this.processWayDataBlock(e, featureWayDoubleDeltaEncoding, linearFeature)) {
                    return false;
                }
                if (e.isPoly() && e.index[0] < 6) continue;
                if (labelPosition != null && wayDataBlock == 0) {
                    e.setLabelPosition(e.points[0] + (float)labelPosition[0], e.points[1] + (float)labelPosition[1]);
                }
                this.mTileProjection.project(e);
                if (!e.tags.containsKey("building") && !this.mTileClipper.clip(e)) continue;
                e.simplify(1.0f, true);
                e.setLayer(layer);
                mapDataSink.process(e);
            }
        }
        return true;
    }

    private int[] readOptionalLabelPosition() {
        int[] labelPosition = new int[2];
        labelPosition[1] = this.mReadBuffer.readSignedInt();
        labelPosition[0] = this.mReadBuffer.readSignedInt();
        return labelPosition;
    }

    private int[][] readZoomTable(SubFileParameter subFileParameter) {
        int rows = subFileParameter.zoomLevelMax - subFileParameter.zoomLevelMin + 1;
        int[][] zoomTable = new int[rows][2];
        int cumulatedNumberOfPois = 0;
        int cumulatedNumberOfWays = 0;
        for (int row = 0; row < rows; ++row) {
            zoomTable[row][0] = cumulatedNumberOfPois += this.mReadBuffer.readUnsignedInt();
            zoomTable[row][1] = cumulatedNumberOfWays += this.mReadBuffer.readUnsignedInt();
        }
        return zoomTable;
    }

    static class TileProjection {
        private static final double COORD_SCALE = 1000000.0;
        long dx;
        long dy;
        double divx;
        double divy;

        TileProjection() {
        }

        void setTile(Tile tile) {
            long x = tile.tileX * Tile.SIZE;
            long y = tile.tileY * Tile.SIZE + Tile.SIZE;
            long mapExtents = Tile.SIZE << tile.zoomLevel;
            this.dx = x - (mapExtents >> 1);
            this.dy = y - (mapExtents >> 1);
            this.divx = 1.8E8 / (double)(mapExtents >> 1);
            this.divy = Math.PI * 2 / (double)(mapExtents >> 1);
        }

        public void projectPoint(int lat, int lon, MapElement out) {
            out.clear();
            out.startPoints();
            out.addPoint(this.projectLon(lon), this.projectLat(lat));
        }

        public float projectLat(double lat) {
            double s = Math.sin(lat * 1.7453292519943295E-8);
            double r = Math.log((1.0 + s) / (1.0 - s));
            return (float)Tile.SIZE - (float)(r / this.divy + (double)this.dy);
        }

        public float projectLon(double lon) {
            return (float)(lon / this.divx - (double)this.dx);
        }

        void project(MapElement e) {
            float[] coords = e.points;
            int[] indices = e.index;
            int inPos = 0;
            int outPos = 0;
            boolean isPoly = e.isPoly();
            for (int len : indices) {
                if (len == 0) continue;
                if (len < 0) break;
                float pLon = 0.0f;
                float pLat = 0.0f;
                int cnt = 0;
                int first = outPos;
                int end = inPos + len;
                while (inPos < end) {
                    float lon = this.projectLon(coords[inPos]);
                    float lat = this.projectLat(coords[inPos + 1]);
                    if (cnt == 0 || lat != pLat || lon != pLon) {
                        coords[outPos++] = pLon = lon;
                        coords[outPos++] = pLat = lat;
                        cnt += 2;
                    }
                    inPos += 2;
                }
                if (isPoly && coords[first] == pLon && coords[first + 1] == pLat) {
                    indices[idx] = (short)(cnt - 2);
                    outPos -= 2;
                    continue;
                }
                indices[idx] = (short)cnt;
            }
            if (e.labelPosition != null) {
                e.labelPosition.x = this.projectLon(e.labelPosition.x);
                e.labelPosition.y = this.projectLat(e.labelPosition.y);
            }
        }
    }
}

