/*
 * Decompiled with CFR 0.152.
 */
package org.oscim.layers.tile.buildings;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.oscim.backend.canvas.Color;
import org.oscim.core.GeometryBuffer;
import org.oscim.utils.ColorsCSS;
import org.oscim.utils.Tessellator;
import org.oscim.utils.geom.GeometryUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class S3DBUtils {
    private static final Logger log = LoggerFactory.getLogger(S3DBUtils.class);
    private static final boolean IMPROVE_RIDGE_CALCULATION = false;
    private static final int SNAP_THRESHOLD = 70;

    private static void addSnapRidgePoint(int id, float[] point, TreeMap<Integer, float[]> ridgePoints) {
        if (point == null) {
            return;
        }
        for (float[] ridPoint : ridgePoints.values()) {
            if (ridPoint == null) {
                log.debug("Ridge point not found!");
                continue;
            }
            if (!(GeometryUtils.distance2D(ridPoint, point) < 70.0)) continue;
            ridgePoints.put(id, ridPoint);
            return;
        }
        ridgePoints.put(id, point);
    }

    public static boolean calcCircleMesh(GeometryBuffer element, float minHeight, float maxHeight, String type) {
        float[] points = element.points;
        int[] index = element.index;
        boolean outerProcessed = false;
        int pointPos = 0;
        for (int i = 0; i < index.length && !outerProcessed && index[i] >= 0; ++i) {
            GeometryBuffer mesh;
            int numSections = index[i] / 2;
            if (numSections < 0) continue;
            outerProcessed = true;
            switch (type) {
                case "onion": {
                    float[][] onionShape = new float[][]{{1.0f, 0.0f, 0.0f}, {0.2f, 0.0f, 0.01f}, {0.875f, 0.0f, 0.1875f}, {1.0f, 0.0f, 0.375f}, {0.875f, 0.0f, 0.5625f}, {0.5f, 0.0f, 0.75f}, {0.2f, 0.0f, 0.8125f}, {0.0f, 0.0f, 1.0f}};
                    mesh = S3DBUtils.initCircleMesh(onionShape, numSections);
                    break;
                }
                default: {
                    float[][] domeShape = new float[][]{{1.0f, 0.0f, 0.0f}, {0.825f, 0.0f, 0.5f}, {0.5f, 0.0f, 0.825f}, {0.0f, 0.0f, 1.0f}};
                    mesh = S3DBUtils.initCircleMesh(domeShape, numSections);
                }
            }
            float centerX = 0.0f;
            float centerY = 0.0f;
            float radius = 0.0f;
            ArrayList<float[]> point3Fs = new ArrayList<float[]>();
            int j = 0;
            while (j < numSections * 2) {
                float x = points[pointPos];
                float y = points[pointPos + 1];
                point3Fs.add(new float[]{x, y, minHeight});
                centerX += x;
                centerY += y;
                j += 2;
                pointPos += 2;
            }
            centerX /= (float)numSections;
            centerY /= (float)numSections;
            for (float[] point3F : point3Fs) {
                float difY;
                float difX = point3F[0] - centerX;
                float tmpR = (float)Math.sqrt(difX * difX + (difY = point3F[1] - centerY) * difY);
                if (!(tmpR > radius)) continue;
                radius = tmpR;
            }
            element.points = mesh.points;
            int numPointsPerSection = element.points.length / (3 * numSections);
            float heightRange = maxHeight - minHeight;
            int j2 = 0;
            for (int k = 0; k < numSections; ++k) {
                float px = ((float[])point3Fs.get(k))[0] - centerX;
                float py = ((float[])point3Fs.get(k))[1] - centerY;
                float phi = (float)Math.atan2(py, px);
                int sectionLimit = (numPointsPerSection + numPointsPerSection * k) * 3;
                boolean first = true;
                while (j2 < sectionLimit) {
                    float r = element.points[j2 + 0] * radius;
                    px = (float)((double)r * Math.cos(phi));
                    py = (float)((double)r * Math.sin(phi));
                    if (!first) {
                        element.points[j2 + 0] = centerX + px;
                        element.points[j2 + 1] = centerY + py;
                        element.points[j2 + 2] = minHeight + element.points[j2 + 2] * heightRange;
                    } else {
                        first = false;
                        element.points[j2 + 0] = ((float[])point3Fs.get(k))[0];
                        element.points[j2 + 1] = ((float[])point3Fs.get(k))[1];
                        element.points[j2 + 2] = minHeight;
                    }
                    j2 += 3;
                }
            }
            element.index = mesh.index;
            element.pointNextPos = element.points.length;
        }
        element.type = GeometryBuffer.GeometryType.TRIS;
        return true;
    }

    public static boolean calcFlatMesh(GeometryBuffer element, float maxHeight) {
        if (Tessellator.tessellate(element, element) == 0) {
            return false;
        }
        float[] points = element.points;
        ArrayList<float[]> point3Fs = new ArrayList<float[]>();
        for (int pointPos = 0; pointPos < points.length; pointPos += 2) {
            float x = points[pointPos];
            float y = points[pointPos + 1];
            point3Fs.add(new float[]{x, y, maxHeight});
        }
        points = new float[3 * point3Fs.size()];
        for (int i = 0; i < point3Fs.size(); ++i) {
            int pPos = 3 * i;
            float[] point3D = (float[])point3Fs.get(i);
            points[pPos + 0] = point3D[0];
            points[pPos + 1] = point3D[1];
            points[pPos + 2] = point3D[2];
        }
        element.points = points;
        element.pointNextPos = element.points.length;
        element.type = GeometryBuffer.GeometryType.TRIS;
        return true;
    }

    public static boolean calcOutlines(GeometryBuffer element, float minHeight, float maxHeight) {
        float[] points = element.points;
        int[] index = element.index;
        element.points = null;
        element.index = null;
        int pointPos = 0;
        for (int i = 0; i < index.length && index[i] >= 0; ++i) {
            int numPoints = index[i] / 2;
            if (numPoints < 0) continue;
            ArrayList<float[]> point3Fs = new ArrayList<float[]>();
            int j = 0;
            while (j < numPoints) {
                float x = points[pointPos];
                float y = points[pointPos + 1];
                point3Fs.add(new float[]{x, y, minHeight});
                point3Fs.add(new float[]{x, y, maxHeight});
                ++j;
                pointPos += 2;
            }
            int[] meshIndex = new int[numPoints * 6];
            for (int j2 = 0; j2 < point3Fs.size(); j2 += 2) {
                int pos = 3 * j2;
                meshIndex[pos + 2] = j2;
                meshIndex[pos + 1] = (j2 + 1) % point3Fs.size();
                meshIndex[pos + 0] = (j2 + 3) % point3Fs.size();
                meshIndex[pos + 5] = (j2 + 3) % point3Fs.size();
                meshIndex[pos + 4] = (j2 + 2) % point3Fs.size();
                meshIndex[pos + 3] = j2;
            }
            float[] meshPoints = new float[point3Fs.size() * 3];
            for (int j3 = 0; j3 < point3Fs.size(); ++j3) {
                int pos = 3 * j3;
                meshPoints[pos + 0] = ((float[])point3Fs.get(j3))[0];
                meshPoints[pos + 1] = ((float[])point3Fs.get(j3))[1];
                meshPoints[pos + 2] = ((float[])point3Fs.get(j3))[2];
            }
            if (element.points == null) {
                element.points = meshPoints;
            } else {
                float[] tmpPoints = element.points;
                element.points = new float[tmpPoints.length + meshPoints.length];
                System.arraycopy(tmpPoints, 0, element.points, 0, tmpPoints.length);
                System.arraycopy(meshPoints, 0, element.points, tmpPoints.length, meshPoints.length);
            }
            if (element.index == null) {
                element.index = meshIndex;
            } else {
                int[] tmpIndex = element.index;
                element.index = new int[tmpIndex.length + meshIndex.length];
                System.arraycopy(tmpIndex, 0, element.index, 0, tmpIndex.length);
                for (int k = 0; k < meshIndex.length; ++k) {
                    element.index[k + tmpIndex.length] = meshIndex[k] + element.pointNextPos / 3;
                }
            }
            element.pointNextPos = element.points.length;
        }
        if (element.points == null) {
            return false;
        }
        element.type = GeometryBuffer.GeometryType.TRIS;
        return true;
    }

    public static boolean calcPyramidalMesh(GeometryBuffer element, float minHeight, float maxHeight) {
        float[] points = element.points;
        int[] index = element.index;
        float[] topPoint = new float[3];
        topPoint[2] = maxHeight;
        int pointPos = 0;
        for (int i = 0; i < index.length && index[i] >= 0 && i <= 0; ++i) {
            int numPoints = index[i] / 2;
            if (numPoints < 0) continue;
            GeometryUtils.center(points, pointPos, numPoints << 1, topPoint);
            ArrayList<float[]> point3Fs = new ArrayList<float[]>();
            int j = 0;
            while (j < numPoints * 2) {
                point3Fs.add(new float[]{points[pointPos], points[pointPos + 1], minHeight});
                j += 2;
                pointPos += 2;
            }
            int[] meshIndex = new int[numPoints * 3];
            for (int j2 = 0; j2 < point3Fs.size(); ++j2) {
                int pos = 3 * j2;
                meshIndex[pos + 0] = j2;
                meshIndex[pos + 1] = (j2 + 1) % point3Fs.size();
                meshIndex[pos + 2] = point3Fs.size();
            }
            point3Fs.add(topPoint);
            float[] meshPoints = new float[point3Fs.size() * 3];
            for (int j3 = 0; j3 < point3Fs.size(); ++j3) {
                int pos = 3 * j3;
                float[] point3D = (float[])point3Fs.get(j3);
                meshPoints[pos + 0] = point3D[0];
                meshPoints[pos + 1] = point3D[1];
                meshPoints[pos + 2] = point3D[2];
            }
            element.points = meshPoints;
            element.index = meshIndex;
            element.pointNextPos = meshPoints.length;
        }
        element.type = GeometryBuffer.GeometryType.TRIS;
        return true;
    }

    public static boolean calcRidgeMesh(GeometryBuffer element, float minHeight, float maxHeight, boolean orientationAcross, boolean isGabled, GeometryBuffer specialParts) {
        float[] points = element.points;
        int[] index = element.index;
        int pointPos = 0;
        for (int i = 0; i < index.length && index[i] >= 0 && i <= 0; ++i) {
            int k;
            int numPoints = index[i] / 2;
            if (numPoints < 0) continue;
            if (numPoints < 4 || !isGabled && orientationAcross) {
                S3DBUtils.calcPyramidalMesh(element, minHeight, maxHeight);
                return true;
            }
            ArrayList<float[]> point3Fs = new ArrayList<float[]>();
            int j = 0;
            while (j < numPoints * 2) {
                float x = points[pointPos];
                float y = points[pointPos + 1];
                point3Fs.add(new float[]{x, y, minHeight});
                j += 2;
                pointPos += 2;
            }
            int groundSize = point3Fs.size();
            ArrayList<Float> lengths = new ArrayList<Float>();
            List<float[]> normVectors = GeometryUtils.normalizedVectors2D(point3Fs, lengths);
            List<Byte> simpleAngles = S3DBUtils.getSimpleAngles(normVectors);
            Integer indexStart = S3DBUtils.getIndexStart(simpleAngles, lengths, orientationAcross);
            int countConcavAngles = 0;
            for (Byte simpleAngle : simpleAngles) {
                if (simpleAngle >= -1) continue;
                ++countConcavAngles;
            }
            if (indexStart == null) {
                if (isGabled) {
                    return S3DBUtils.calcSimpleGabledMesh(element, minHeight, maxHeight, orientationAcross, specialParts);
                }
                return S3DBUtils.calcPyramidalMesh(element, minHeight, maxHeight);
            }
            List<float[]> bisections = S3DBUtils.getBisections(normVectors);
            ArrayList<float[]> intersections = new ArrayList<float[]>();
            for (int k2 = 0; k2 < groundSize; ++k2) {
                int nextTurn = S3DBUtils.getIndexNextTurn(k2, simpleAngles);
                float[] pA = (float[])point3Fs.get(nextTurn);
                float[] pB = (float[])point3Fs.get(k2);
                intersections.add(GeometryUtils.intersectionLines2D(pA, bisections.get(nextTurn), pB, bisections.get(k2)));
            }
            TreeMap<Integer, float[]> ridgePoints = new TreeMap<Integer, float[]>();
            TreeMap<Integer, float[]> ridgeLines = new TreeMap<Integer, float[]>();
            HashSet<Integer> gablePoints = new HashSet<Integer>();
            Integer currentRidgeInd = null;
            boolean isOdd = false;
            for (int k3 = 0; k3 < groundSize; ++k3) {
                int shift = (k3 + indexStart) % groundSize;
                byte direction = simpleAngles.get(shift);
                if (direction == 0) continue;
                if (direction < 0) {
                    Integer opposite;
                    Integer indexNext;
                    float[] positionRidgeA = null;
                    float[] positionRidgeB = null;
                    Integer indexPrevious = S3DBUtils.getIndexPreviousConvexTurn(shift, simpleAngles);
                    Integer indexPrevious2 = S3DBUtils.getIndexPreviousConvexTurn(indexPrevious == null ? shift - 1 : indexPrevious, simpleAngles);
                    if (indexPrevious != null && indexPrevious2 != null) {
                        if (!ridgeLines.containsKey(indexPrevious2)) {
                            ridgeLines.put(indexPrevious2, normVectors.get(indexPrevious));
                        }
                        positionRidgeA = (float[])intersections.get(indexPrevious2);
                        currentRidgeInd = indexPrevious2;
                        if (isGabled) {
                            positionRidgeA = GeometryUtils.intersectionLines2D(positionRidgeA, (float[])ridgeLines.get(indexPrevious2), (float[])point3Fs.get(indexPrevious2), normVectors.get(indexPrevious2));
                            gablePoints.add(indexPrevious2);
                        }
                        ridgePoints.put(indexPrevious2, positionRidgeA);
                        gablePoints.remove(indexPrevious);
                        ridgePoints.remove(indexPrevious);
                        ridgeLines.remove(indexPrevious);
                    }
                    Integer indexNext2 = S3DBUtils.getIndexNextConvexTurn((indexNext = S3DBUtils.getIndexNextConvexTurn(shift, simpleAngles)) == null ? shift + 1 : indexNext, simpleAngles);
                    if (indexNext != null && indexNext2 != null) {
                        if (ridgePoints.get(indexNext) == null) {
                            if (!ridgeLines.containsKey(indexNext)) {
                                ridgeLines.put(indexNext, normVectors.get(indexNext2));
                            }
                            positionRidgeB = (float[])intersections.get(indexNext);
                            if (isGabled) {
                                positionRidgeB = GeometryUtils.intersectionLines2D(positionRidgeB, (float[])ridgeLines.get(indexNext), (float[])point3Fs.get(indexNext), normVectors.get(indexNext));
                                gablePoints.add(indexNext);
                            }
                            ridgePoints.put(indexNext, positionRidgeB);
                        } else {
                            positionRidgeB = (float[])ridgePoints.get(indexNext);
                        }
                    }
                    if (positionRidgeA == null || positionRidgeB == null) {
                        if (positionRidgeA == null && positionRidgeB == null && currentRidgeInd != null) {
                            positionRidgeA = ridgePoints.get(currentRidgeInd);
                        }
                        if (positionRidgeA != null && positionRidgeB == null) {
                            positionRidgeA = GeometryUtils.intersectionLines2D(positionRidgeA, (float[])ridgeLines.get(currentRidgeInd), (float[])point3Fs.get(shift), bisections.get(shift));
                            currentRidgeInd = shift;
                            S3DBUtils.addSnapRidgePoint(shift, positionRidgeA, ridgePoints);
                            ridgeLines.put(shift, normVectors.get(shift));
                            isOdd = false;
                            continue;
                        }
                        if (positionRidgeA == null && positionRidgeB != null) {
                            positionRidgeA = GeometryUtils.intersectionLines2D(positionRidgeB, (float[])ridgeLines.get(indexNext), (float[])point3Fs.get(shift), bisections.get(shift));
                            S3DBUtils.addSnapRidgePoint(shift, positionRidgeA, ridgePoints);
                            currentRidgeInd = null;
                            isOdd = false;
                            continue;
                        }
                        log.debug("Should never happen, because positionRidge wouldn't be null then");
                        currentRidgeInd = null;
                        continue;
                    }
                    if (currentRidgeInd == null || indexNext == null || ridgeLines.get(currentRidgeInd) == null || ridgeLines.get(indexNext) == null) {
                        log.debug("Concave shape not calculated correctly: " + element.toString());
                        currentRidgeInd = null;
                        continue;
                    }
                    float[] intersection = GeometryUtils.intersectionLines2D(positionRidgeA, (float[])ridgeLines.get(currentRidgeInd), positionRidgeB, (float[])ridgeLines.get(indexNext));
                    S3DBUtils.addSnapRidgePoint(shift, intersection, ridgePoints);
                    if (countConcavAngles == 1 && (opposite = S3DBUtils.getIndexNextConvexTurn(indexNext2, simpleAngles)) != null) {
                        if (isGabled) {
                            gablePoints.remove(opposite);
                        }
                        ridgePoints.put(opposite, intersection);
                    }
                    currentRidgeInd = null;
                    isOdd = false;
                    continue;
                }
                if (isOdd) {
                    isOdd = false;
                    continue;
                }
                if (simpleAngles.get(shift) > 1) {
                    isOdd = true;
                }
                if (ridgePoints.containsKey(shift) && ridgeLines.containsKey(shift)) {
                    currentRidgeInd = shift;
                    continue;
                }
                if (currentRidgeInd != null) {
                    float[] intersection;
                    if (isGabled && direction > 1) {
                        if (ridgePoints.get(currentRidgeInd) == null) {
                            log.debug("Gabled intersection calc failed");
                            currentRidgeInd = null;
                            continue;
                        }
                        intersection = GeometryUtils.intersectionLines2D((float[])ridgePoints.get(currentRidgeInd), (float[])ridgeLines.get(currentRidgeInd), (float[])point3Fs.get(shift), normVectors.get(shift));
                        if (intersection == null) {
                            log.debug("Gabled intersection calc failed");
                            currentRidgeInd = null;
                            continue;
                        }
                        gablePoints.add(shift);
                        ridgePoints.put(shift, intersection);
                    } else {
                        intersection = GeometryUtils.intersectionLines2D((float[])ridgePoints.get(currentRidgeInd), (float[])ridgeLines.get(currentRidgeInd), (float[])point3Fs.get(shift), bisections.get(shift));
                        S3DBUtils.addSnapRidgePoint(shift, intersection, ridgePoints);
                    }
                    if (isOdd) {
                        currentRidgeInd = null;
                        continue;
                    }
                    ridgeLines.put(shift, normVectors.get(shift));
                    currentRidgeInd = shift;
                    continue;
                }
                Integer indexNext = S3DBUtils.getIndexNextConvexTurn(shift, simpleAngles);
                if (indexNext == null) continue;
                if (!ridgeLines.containsKey(shift)) {
                    ridgeLines.put(shift, normVectors.get(indexNext));
                }
                currentRidgeInd = shift;
                float[] ridgePos = (float[])intersections.get(shift);
                if (isGabled) {
                    ridgePos = GeometryUtils.intersectionLines2D(ridgePos, (float[])ridgeLines.get(currentRidgeInd), (float[])point3Fs.get(shift), normVectors.get(shift));
                    gablePoints.add(shift);
                }
                S3DBUtils.addSnapRidgePoint(shift, ridgePos, ridgePoints);
            }
            if (ridgePoints.isEmpty()) {
                S3DBUtils.calcPyramidalMesh(element, minHeight, maxHeight);
                return true;
            }
            Iterator ridgeIt = ridgePoints.entrySet().iterator();
            while (ridgeIt.hasNext()) {
                boolean isIn;
                Map.Entry ridgeEntry = ridgeIt.next();
                Integer key = (Integer)ridgeEntry.getKey();
                if (ridgeEntry.getValue() == null) {
                    log.debug("Ridge calculation failed at point " + key);
                    ridgeIt.remove();
                    continue;
                }
                if (isGabled && simpleAngles.get(key) >= 0 || (isIn = GeometryUtils.pointInPoly(((float[])ridgeEntry.getValue())[0], ((float[])ridgeEntry.getValue())[1], points, points.length, 0))) continue;
                if (isGabled) {
                    return S3DBUtils.calcSimpleGabledMesh(element, minHeight, maxHeight, orientationAcross, specialParts);
                }
                return S3DBUtils.calcFlatMesh(element, minHeight);
            }
            int ridgePointSize = ridgePoints.size();
            float[] meshPoints = new float[(groundSize + ridgePointSize) * 3];
            ArrayList<Integer> meshVarIndex = new ArrayList<Integer>();
            ArrayList<Integer> meshPartVarIndex = null;
            if (isGabled && specialParts != null) {
                meshPartVarIndex = new ArrayList<Integer>();
            }
            for (int k4 = 0; k4 < groundSize; ++k4) {
                float[] p = (float[])point3Fs.get(k4);
                int ridgePointIndex1 = k4;
                while (!ridgePoints.containsKey(ridgePointIndex1)) {
                    ridgePointIndex1 = (ridgePointIndex1 + groundSize - 1) % groundSize;
                }
                int ridgeIndex1 = ridgePoints.headMap(ridgePointIndex1).size();
                if (meshPartVarIndex != null && gablePoints.contains(ridgePointIndex1) && S3DBUtils.getIndexNextTurn(ridgePointIndex1, simpleAngles).equals(S3DBUtils.getIndexNextTurn(k4, simpleAngles))) {
                    meshPartVarIndex.add(k4);
                    meshPartVarIndex.add((k4 + 1) % groundSize);
                    meshPartVarIndex.add(ridgeIndex1 + groundSize);
                } else {
                    meshVarIndex.add(k4);
                    meshVarIndex.add((k4 + 1) % groundSize);
                    meshVarIndex.add(ridgeIndex1 + groundSize);
                }
                int ridgePointIndex2 = (k4 + 1) % groundSize;
                while (!ridgePoints.containsKey(ridgePointIndex2)) {
                    ridgePointIndex2 = (ridgePointIndex2 + groundSize - 1) % groundSize;
                }
                if (ridgePointIndex2 != ridgePointIndex1) {
                    int ridgeIndex2 = ridgePoints.headMap(ridgePointIndex2).size();
                    meshVarIndex.add(ridgeIndex1 + groundSize);
                    meshVarIndex.add((k4 + 1) % groundSize);
                    meshVarIndex.add(ridgeIndex2 + groundSize);
                }
                meshPoints[3 * k4 + 0] = p[0];
                meshPoints[3 * k4 + 1] = p[1];
                meshPoints[3 * k4 + 2] = p[2];
            }
            if (ridgePointSize > 2) {
                HashSet<Integer> ridgeSkipFaceIndex = new HashSet<Integer>();
                boolean isTessellateAble = true;
                block9: for (int k5 = 0; k5 < groundSize; ++k5) {
                    if (!isTessellateAble || ridgePoints.get(k5) == null) continue;
                    Integer middle = null;
                    for (int m = k5 + 1; m <= k5 + groundSize; ++m) {
                        int secIndex = m % groundSize;
                        if (ridgePoints.get(secIndex) == null) continue;
                        if (middle == null) {
                            middle = secIndex;
                            continue;
                        }
                        float isClockwise = GeometryUtils.isTrisClockwise(ridgePoints.get(k5), ridgePoints.get(middle), ridgePoints.get(secIndex));
                        if ((double)Math.abs(isClockwise) < 0.001) {
                            ridgeSkipFaceIndex.add(middle);
                            if (Arrays.equals(ridgePoints.get(k5), ridgePoints.get(secIndex))) {
                                ridgeSkipFaceIndex.add(k5);
                            }
                        }
                        if (!(isClockwise > 0.0f)) continue block9;
                        continue block9;
                    }
                }
                int faceLength = ridgePointSize - ridgeSkipFaceIndex.size();
                if (isTessellateAble && faceLength > 0) {
                    float[] gbPoints = new float[2 * faceLength];
                    int k6 = 0;
                    ArrayList<Integer> faceIndex = new ArrayList<Integer>();
                    for (int m = 0; m < groundSize; ++m) {
                        float[] point = ridgePoints.get(m);
                        if (ridgeSkipFaceIndex.contains(m) || point == null) continue;
                        faceIndex.add(m);
                        gbPoints[2 * k6] = point[0];
                        gbPoints[2 * k6 + 1] = point[1];
                        ++k6;
                    }
                    GeometryBuffer buffer = new GeometryBuffer(gbPoints, new int[]{2 * faceLength});
                    if (Tessellator.tessellate(buffer, buffer) != 0) {
                        for (int ind : buffer.index) {
                            meshVarIndex.add(ridgePoints.headMap((Integer)faceIndex.get(ind)).size() + groundSize);
                        }
                    } else {
                        if (isGabled) {
                            return S3DBUtils.calcSimpleGabledMesh(element, minHeight, maxHeight, orientationAcross, specialParts);
                        }
                        return S3DBUtils.calcFlatMesh(element, minHeight);
                    }
                }
            }
            int[] meshIndex = new int[meshVarIndex.size()];
            for (k = 0; k < meshIndex.length; ++k) {
                meshIndex[k] = (Integer)meshVarIndex.get(k);
            }
            int l = 0;
            for (k = 0; k < groundSize; ++k) {
                float[] tmp = ridgePoints.get(k);
                if (tmp == null) continue;
                float[] p = new float[]{tmp[0], tmp[1], maxHeight};
                int ppos = 3 * (l + groundSize);
                meshPoints[ppos + 0] = p[0];
                meshPoints[ppos + 1] = p[1];
                meshPoints[ppos + 2] = p[2];
                ++l;
            }
            if (specialParts != null && meshPartVarIndex != null) {
                int[] meshPartsIndex = new int[meshPartVarIndex.size()];
                for (int k7 = 0; k7 < meshPartsIndex.length; ++k7) {
                    meshPartsIndex[k7] = (Integer)meshPartVarIndex.get(k7);
                }
                specialParts.points = meshPoints;
                specialParts.index = meshPartsIndex;
                specialParts.pointNextPos = meshPoints.length;
                specialParts.type = GeometryBuffer.GeometryType.TRIS;
            }
            element.points = meshPoints;
            element.index = meshIndex;
            element.pointNextPos = meshPoints.length;
            element.type = GeometryBuffer.GeometryType.TRIS;
        }
        return element.isTris();
    }

    private static boolean calcSimpleGabledMesh(GeometryBuffer element, float minHeight, float maxHeight, boolean orientationAcross, GeometryBuffer specialParts) {
        float[] points = element.points;
        int[] index = element.index;
        int pointPos = 0;
        for (int i = 0; i < index.length && index[i] >= 0 && i <= 0; ++i) {
            int numPoints = index[i] / 2;
            if (numPoints < 0) continue;
            if (numPoints < 4) {
                S3DBUtils.calcPyramidalMesh(element, minHeight, maxHeight);
                return true;
            }
            ArrayList<float[]> point3Fs = new ArrayList<float[]>();
            int j = 0;
            while (j < numPoints * 2) {
                float x = points[pointPos];
                float y = points[pointPos + 1];
                point3Fs.add(new float[]{x, y, minHeight});
                j += 2;
                pointPos += 2;
            }
            int groundSize = point3Fs.size();
            ArrayList<Float> lengths = new ArrayList<Float>();
            List<float[]> normVectors = GeometryUtils.normalizedVectors2D(point3Fs, lengths);
            List<Byte> simpleAngles = S3DBUtils.getSimpleAngles(normVectors);
            int indexStart = S3DBUtils.getIndicesLongestSide(simpleAngles, lengths, null)[0];
            if (orientationAcross) {
                Integer tmp = S3DBUtils.getIndexPreviousConvexTurn(indexStart, simpleAngles);
                if (tmp == null) {
                    tmp = S3DBUtils.getIndexNextTurn(indexStart, simpleAngles);
                }
                indexStart = tmp;
            }
            float[] vL = normVectors.get(indexStart);
            float[] pL = (float[])point3Fs.get(indexStart);
            float[] splitLinePoint = null;
            float maxDist = 0.0f;
            for (float[] point : point3Fs) {
                float curDist = GeometryUtils.distancePointLine2D(point, pL, vL);
                if (!(curDist > maxDist)) continue;
                maxDist = curDist;
                splitLinePoint = point;
            }
            maxDist = Math.signum(GeometryUtils.isTrisClockwise(pL, GeometryUtils.sumVec(pL, vL), splitLinePoint)) * (maxDist / 2.0f);
            float[] normL = new float[]{-vL[1], vL[0]};
            normL = GeometryUtils.scale(normL, (float)((double)maxDist / Math.sqrt(GeometryUtils.dotProduct(normL, normL))));
            splitLinePoint = GeometryUtils.sumVec(pL, normL);
            float degreeNormL = (float)Math.atan2(normL[0], -normL[1]) * 57.295776f;
            int sideChange = 0;
            ArrayList<Object> elementPoints1 = new ArrayList<Object>();
            ArrayList<Object> elementPoints2 = new ArrayList<Object>();
            float[] secSplitPoint = GeometryUtils.sumVec(splitLinePoint, vL);
            float sideLastPoint = Math.signum(GeometryUtils.isTrisClockwise(splitLinePoint, secSplitPoint, (float[])point3Fs.get(groundSize - 1)));
            degreeNormL = sideLastPoint > 0.0f ? degreeNormL : (degreeNormL + 180.0f) % 360.0f;
            ArrayList<Integer> intersection1 = new ArrayList<Integer>();
            ArrayList<Integer> intersection2 = new ArrayList<Integer>();
            for (int k = 0; k < groundSize; ++k) {
                float sideCurPoint = Math.signum(GeometryUtils.isTrisClockwise(splitLinePoint, secSplitPoint, (float[])point3Fs.get(k)));
                if (sideCurPoint != sideLastPoint) {
                    if (sideChange > 2) {
                        return S3DBUtils.calcFlatMesh(element, minHeight);
                    }
                    int indexPrev = (k + groundSize - 1) % groundSize;
                    float[] intersection = GeometryUtils.intersectionLines2D(splitLinePoint, vL, (float[])point3Fs.get(indexPrev), normVectors.get(indexPrev));
                    elementPoints1.add(intersection);
                    elementPoints2.add(intersection);
                    intersection1.add(elementPoints1.size() - 1);
                    intersection2.add(elementPoints2.size() - 1);
                    ++sideChange;
                }
                if (sideChange % 2 == 0) {
                    elementPoints1.add(point3Fs.get(k));
                } else {
                    elementPoints2.add(point3Fs.get(k));
                }
                sideLastPoint = sideCurPoint;
            }
            GeometryBuffer geoEle1 = new GeometryBuffer(elementPoints1.size(), 1);
            for (int k = 0; k < elementPoints1.size(); ++k) {
                geoEle1.points[2 * k] = ((float[])elementPoints1.get(k))[0];
                geoEle1.points[2 * k + 1] = ((float[])elementPoints1.get(k))[1];
            }
            geoEle1.index[0] = geoEle1.points.length;
            geoEle1.pointNextPos = geoEle1.points.length;
            GeometryBuffer geoEle2 = new GeometryBuffer(elementPoints2.size(), 1);
            for (int k = 0; k < elementPoints2.size(); ++k) {
                geoEle2.points[2 * k] = ((float[])elementPoints2.get(k))[0];
                geoEle2.points[2 * k + 1] = ((float[])elementPoints2.get(k))[1];
            }
            geoEle2.index[0] = geoEle2.points.length;
            geoEle2.pointNextPos = geoEle2.points.length;
            GeometryBuffer specialParts1 = new GeometryBuffer(geoEle1);
            GeometryBuffer specialParts2 = new GeometryBuffer(geoEle2);
            if (!S3DBUtils.calcSkillionMesh(geoEle1, minHeight, maxHeight, degreeNormL, specialParts1) || !S3DBUtils.calcSkillionMesh(geoEle2, minHeight, maxHeight, degreeNormL + 180.0f, specialParts2)) {
                return false;
            }
            for (Integer integer : intersection1) {
                geoEle1.points[integer.intValue() * 3 + 2] = maxHeight;
                specialParts1.points[6 * integer.intValue() + 5] = maxHeight;
            }
            for (Integer integer : intersection2) {
                geoEle2.points[integer.intValue() * 3 + 2] = maxHeight;
                specialParts2.points[6 * integer.intValue() + 5] = maxHeight;
            }
            S3DBUtils.mergeMeshGeometryBuffer(geoEle1, geoEle2, element);
            S3DBUtils.mergeMeshGeometryBuffer(specialParts1, specialParts2, specialParts);
            return true;
        }
        return false;
    }

    public static boolean calcSkillionMesh(GeometryBuffer element, float minHeight, float maxHeight, float roofDegree, GeometryBuffer specialParts) {
        float[] points = element.points;
        int[] index = element.index;
        int pointPos = 0;
        for (int i = 0; i < index.length && index[i] >= 0 && i <= 0; ++i) {
            float[] normal;
            int numPoints = index[i] / 2;
            if (numPoints < 0) continue;
            ArrayList<float[]> point3Fs = new ArrayList<float[]>();
            int j = 0;
            while (j < numPoints * 2) {
                float x = points[pointPos];
                float y = points[pointPos + 1];
                point3Fs.add(new float[]{x, y, minHeight});
                j += 2;
                pointPos += 2;
            }
            boolean hasOutlines = S3DBUtils.calcOutlines(specialParts, minHeight, maxHeight);
            float[] min1 = null;
            float[] min2 = null;
            float[] max1 = null;
            float[] max2 = null;
            float minDif2 = Float.MAX_VALUE;
            float minDif1 = Float.MAX_VALUE;
            float maxDif2 = 0.0f;
            float maxDif1 = 0.0f;
            if (!S3DBUtils.calcFlatMesh(element, maxHeight)) continue;
            roofDegree = ((float)Math.PI / 180 * roofDegree + (float)Math.PI * 2) % ((float)Math.PI * 2);
            float[] vRidge = new float[]{(float)Math.sin(roofDegree), (float)(-Math.cos(roofDegree))};
            vRidge = GeometryUtils.scale(vRidge, 1.0E8f);
            for (int k = 0; k < point3Fs.size(); ++k) {
                float[] point = (float[])point3Fs.get(k);
                float vx = vRidge[0] - point[0];
                float vy = vRidge[1] - point[1];
                float currentDiff = (float)Math.sqrt(vx * vx + vy * vy);
                if (max1 == null || currentDiff > maxDif1) {
                    if (max1 != null) {
                        max2 = max1;
                        maxDif2 = maxDif1;
                    }
                    max1 = point;
                    maxDif1 = currentDiff;
                } else if (max2 == null || currentDiff > maxDif2) {
                    max2 = point;
                    maxDif2 = currentDiff;
                }
                if (min1 == null || currentDiff < minDif1) {
                    if (min1 != null) {
                        min2 = min1;
                        minDif2 = minDif1;
                    }
                    min1 = point;
                    minDif1 = currentDiff;
                    continue;
                }
                if (min2 != null && !(currentDiff < minDif2)) continue;
                min2 = point;
                minDif2 = currentDiff;
            }
            if (min1 == max1) {
                return false;
            }
            float[] zVector = new float[]{0.0f, 0.0f, 1.0f};
            min1[2] = minHeight;
            max1[2] = maxHeight;
            if (Math.abs(minDif2 - minDif1) < Math.abs(maxDif2 - maxDif1)) {
                min2[2] = minHeight;
                normal = GeometryUtils.normalOfPlane(min1, max1, min2);
            } else {
                max2[2] = maxHeight;
                normal = GeometryUtils.normalOfPlane(min1, max1, max2);
            }
            for (int k = 0; k < point3Fs.size(); ++k) {
                float[] intersection = GeometryUtils.intersectionLinePlane((float[])point3Fs.get(k), zVector, min1, normal);
                if (intersection == null) {
                    return false;
                }
                intersection[2] = intersection[2] > 2.0f * maxHeight ? maxHeight : (intersection[2] < minHeight ? minHeight : intersection[2]);
                element.points[3 * k + 2] = intersection[2];
                if (!hasOutlines) continue;
                specialParts.points[6 * k + 5] = intersection[2];
            }
            return true;
        }
        return false;
    }

    private static List<float[]> getBisections(List<float[]> normVectors) {
        int size = normVectors.size();
        ArrayList<float[]> bisections = new ArrayList<float[]>();
        for (int k = 0; k < size; ++k) {
            float[] vBC = normVectors.get((k + size - 1) % size);
            float[] vBA = normVectors.get(k);
            vBC = Arrays.copyOf(vBC, vBC.length);
            vBC[0] = -vBC[0];
            vBC[1] = -vBC[1];
            bisections.add(GeometryUtils.bisectionNorm2D(vBC, vBA));
        }
        return bisections;
    }

    public static int getColor(String color, Color.HSV hsv, boolean relative) {
        int c;
        if ("transparent".equals(color)) {
            return Color.get(0, 1, 1, 1);
        }
        if (color.charAt(0) == '#') {
            c = Color.parseColor(color, -16711681);
        } else {
            Integer css = ColorsCSS.get(color);
            if (css == null) {
                log.debug("unknown color:{}", (Object)color);
                c = -16711681;
            } else {
                c = css;
            }
        }
        return hsv.mod(c, relative);
    }

    private static Integer getIndexNextConvexTurn(int index, List<Byte> simpleAngles) {
        for (int i = index + 1; i < simpleAngles.size() + index; ++i) {
            int iMod = i % simpleAngles.size();
            if (simpleAngles.get(iMod) > 0) {
                return iMod;
            }
            if (simpleAngles.get(iMod) >= 0) continue;
            return null;
        }
        return (index + 1) % simpleAngles.size();
    }

    private static Integer getIndexNextTurn(int index, List<Byte> simpleAngles) {
        for (int i = index + 1; i < simpleAngles.size() + index; ++i) {
            int iMod = i % simpleAngles.size();
            if (simpleAngles.get(iMod) == 0) continue;
            return iMod;
        }
        return (index + 1) % simpleAngles.size();
    }

    private static Integer getIndexPreviousConvexTurn(int index, List<Byte> simpleAngles) {
        for (int i = simpleAngles.size() + index - 1; i >= 0; --i) {
            int iMod = i % simpleAngles.size();
            if (simpleAngles.get(iMod) > 0) {
                return iMod;
            }
            if (simpleAngles.get(iMod) >= 0) continue;
            return null;
        }
        return (simpleAngles.size() + index - 1) % simpleAngles.size();
    }

    private static Integer getIndexStart(List<Byte> simpleAngles, List<Float> lengths, boolean directionAcross) {
        int[] iLongSide;
        int i;
        int size = simpleAngles.size();
        Integer indexStart = null;
        Integer concaveStart = null;
        for (i = 0; i < size && (indexStart == null || concaveStart == null); ++i) {
            if (indexStart == null && simpleAngles.get(i) > 1) {
                indexStart = i;
                continue;
            }
            if (concaveStart != null || simpleAngles.get(i) >= -1) continue;
            concaveStart = i;
        }
        if (indexStart == null) {
            return null;
        }
        if (concaveStart != null) {
            for (i = concaveStart.intValue(); i < size + indexStart; ++i) {
                if (simpleAngles.get(i % size) >= 0) continue;
                return i % size;
            }
        }
        indexStart = simpleAngles.get((iLongSide = S3DBUtils.getIndicesLongestSide(simpleAngles, lengths, indexStart))[1]) < 2 ? S3DBUtils.getIndexPreviousConvexTurn(iLongSide[0], simpleAngles) : Integer.valueOf(iLongSide[1]);
        if (directionAcross) {
            return iLongSide[0];
        }
        return indexStart;
    }

    private static int[] getIndicesLongestSide(List<Byte> simpleAngles, List<Float> lengths, Integer indexStart) {
        int[] iLongSide = new int[2];
        int size = simpleAngles.size();
        if (indexStart == null) {
            for (int i = 0; i < size; ++i) {
                if (simpleAngles.get(i) <= 0) continue;
                indexStart = i;
                break;
            }
        }
        float longestSideLength = 0.0f;
        float currentLength = 0.0f;
        int indexCurrentSide = indexStart;
        int loopSize = size + indexStart;
        for (int i = indexStart.intValue(); i < loopSize; ++i) {
            if (i >= size) {
                i -= size;
                loopSize -= size;
            }
            if (simpleAngles.get(i) != 0) {
                currentLength = lengths.get(i).floatValue();
                indexCurrentSide = i;
            } else {
                currentLength += lengths.get(i).floatValue();
            }
            if (!(currentLength > longestSideLength)) continue;
            longestSideLength = currentLength;
            iLongSide[0] = indexCurrentSide;
            iLongSide[1] = (i + 1) % size;
        }
        return iLongSide;
    }

    public static int getMaterialColor(String material, Color.HSV hsv, boolean relative) {
        int c;
        if (material.charAt(0) == '#') {
            c = Color.parseColor(material, -16711681);
        } else {
            switch (material) {
                case "roof_tiles": {
                    c = Color.get(216, 167, 111);
                    break;
                }
                case "tile": {
                    c = Color.get(216, 167, 111);
                    break;
                }
                case "concrete": 
                case "cement_block": {
                    c = Color.get(210, 212, 212);
                    break;
                }
                case "metal": {
                    c = -4144960;
                    break;
                }
                case "tar_paper": {
                    c = -6907496;
                    break;
                }
                case "eternit": {
                    c = Color.get(216, 167, 111);
                    break;
                }
                case "tin": {
                    c = -4144960;
                    break;
                }
                case "asbestos": {
                    c = Color.get(160, 152, 141);
                    break;
                }
                case "glass": {
                    c = Color.fade(Color.get(130, 224, 255), 0.6f);
                    break;
                }
                case "slate": {
                    c = -10462880;
                    break;
                }
                case "zink": {
                    c = Color.get(180, 180, 180);
                    break;
                }
                case "gravel": {
                    c = Color.get(170, 130, 80);
                    break;
                }
                case "copper": {
                    c = Color.get(150, 200, 130);
                    break;
                }
                case "wood": {
                    c = Color.get(170, 130, 80);
                    break;
                }
                case "grass": {
                    c = -11490736;
                    break;
                }
                case "stone": {
                    c = Color.get(206, 207, 181);
                    break;
                }
                case "plaster": {
                    c = Color.get(236, 237, 181);
                    break;
                }
                case "brick": {
                    c = Color.get(255, 217, 191);
                    break;
                }
                case "stainless_steel": {
                    c = Color.get(153, 157, 160);
                    break;
                }
                case "gold": {
                    c = -10496;
                    break;
                }
                default: {
                    c = -16711681;
                    log.debug("unknown material:{}", (Object)material);
                }
            }
        }
        return hsv.mod(c, relative);
    }

    private static List<Byte> getSimpleAngles(List<float[]> normVectors) {
        int size = normVectors.size();
        ArrayList<Byte> simpAngls = new ArrayList<Byte>();
        float tmpAnlgeSum = 0.0f;
        float threshold = 0.2617994f;
        for (int k = 0; k < size; ++k) {
            float[] v2 = normVectors.get(k);
            float[] v1 = normVectors.get((k - 1 + size) % size);
            float val = v1[0] * v2[0] + v1[1] * v2[1];
            float angle = (float)Math.acos(Math.abs(val) > 1.0f ? (double)Math.signum(val) : (double)val);
            byte simpAngle = (byte)Math.signum(v1[0] * v2[1] - v1[1] * v2[0]);
            if (angle > 1.5707964f - threshold) {
                simpAngle = (byte)(simpAngle * 2);
                tmpAnlgeSum = 0.0f;
            } else if (angle < threshold) {
                if (Math.abs(tmpAnlgeSum += (float)simpAngle * angle) > threshold) {
                    simpAngle = (byte)Math.signum(tmpAnlgeSum);
                    tmpAnlgeSum = 0.0f;
                } else {
                    simpAngle = 0;
                }
            } else {
                tmpAnlgeSum = 0.0f;
            }
            simpAngls.add(simpAngle);
        }
        return simpAngls;
    }

    private static GeometryBuffer initCircleMesh(float[][] circleShape, int numSections) {
        int indexSize = numSections * (circleShape.length - 1) * 2 * 3;
        int[] meshIndex = new int[indexSize];
        int meshSize = numSections * circleShape.length;
        float[] meshPoints = new float[meshSize * 3];
        for (int i = 0; i < numSections; ++i) {
            for (int j = 0; j < circleShape.length; ++j) {
                int pPos = 3 * (i * circleShape.length + j);
                meshPoints[pPos + 0] = circleShape[j][0];
                meshPoints[pPos + 1] = circleShape[j][1];
                meshPoints[pPos + 2] = circleShape[j][2];
                if (j == circleShape.length - 1) continue;
                int iPos = 6 * (i * (circleShape.length - 1) + j);
                meshIndex[iPos + 2] = (pPos /= 3) + 0;
                meshIndex[iPos + 1] = pPos + 1;
                meshIndex[iPos + 0] = (pPos + circleShape.length) % meshSize;
                meshIndex[iPos + 5] = pPos + 1;
                meshIndex[iPos + 4] = (pPos + circleShape.length + 1) % meshSize;
                meshIndex[iPos + 3] = (pPos + circleShape.length) % meshSize;
            }
        }
        return new GeometryBuffer(meshPoints, meshIndex);
    }

    private static void mergeMeshGeometryBuffer(GeometryBuffer gb1, GeometryBuffer gb2, GeometryBuffer out) {
        if (!gb1.isTris() || !gb2.isTris()) {
            return;
        }
        int gb1PointSize = gb1.points.length;
        float[] mergedPoints = new float[gb1PointSize + gb2.points.length];
        System.arraycopy(gb1.points, 0, mergedPoints, 0, gb1PointSize);
        System.arraycopy(gb2.points, 0, mergedPoints, gb1PointSize, gb2.points.length);
        out.points = mergedPoints;
        out.pointNextPos = mergedPoints.length;
        int gb1IndexSize = gb1.index.length;
        int[] mergedIndices = new int[gb1IndexSize + gb2.index.length];
        System.arraycopy(gb1.index, 0, mergedIndices, 0, gb1IndexSize);
        gb1PointSize /= 3;
        for (int k = 0; k < gb2.index.length; ++k) {
            mergedIndices[gb1IndexSize + k] = gb2.index[k] + gb1PointSize;
        }
        out.index = mergedIndices;
        out.type = gb1.type;
    }

    private S3DBUtils() {
    }
}

