package org.mbari.vcr4j.sharktopoda.client.localization;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.mbari.vcr4j.sharktopoda.client.gson.DurationConverter;
import org.mbari.vcr4j.util.StringUtils;
import org.zeromq.SocketType;
import org.zeromq.ZContext;
import org.zeromq.ZMQ;
import org.zeromq.ZMQ.Socket;
import org.zeromq.ZMQException;

import java.time.Duration;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * @author Brian Schlining
 * @since 2020-02-11T17:00:00
 */
public class IO {

    private static final System.Logger log = System.getLogger(IO.class.getName());
    private ZContext context = new ZContext();
    private final int incomingPort;
    private final int outgoingPort;
    private final LocalizationController controller;
    private final SelectionController selectionController;
    private final Thread outgoingThread;
    private final Thread incomingThread;
    private volatile boolean ok = true;
    private LinkedBlockingQueue<Message> queue = new LinkedBlockingQueue<>();
    private final Gson gson = new GsonBuilder()
            .setDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'")
            .registerTypeAdapter(Duration .class, new DurationConverter())
            .create();
    private final String sourceId = StringUtils.randomString(10);

    public IO(int incomingPort,
              int outgoingPort,
              String incomingTopic,
              String outgoingTopic,
              LocalizationController controller) {

        this.incomingPort = incomingPort;
        this.outgoingPort = outgoingPort;
        this.controller = controller;
        this.controller.getOutgoing()
                .ofType(Message.class)
                .subscribe(lcl -> queue.offer(lcl));
        this.selectionController = new SelectionController(controller);

        // publisher
        outgoingThread = new Thread(() -> {
            String address = "tcp://*:" + outgoingPort;
            log.log(System.Logger.Level.INFO, () -> "ZeroMQ Publishing to " + address + " using topic '" + outgoingTopic + "'");
            Socket publisher = context.createSocket(SocketType.PUB);
            publisher.bind(address);
            // This sleep is critical to give ZMQ time to bind and setup
            // Otherwise messages will not be sent.
            try {
                Thread.sleep(1000);
            }
            catch (InterruptedException e) {
                log.log(System.Logger.Level.WARNING, "ZeroMQ publisher thread was interrupted", e);
            }
            while (ok && !Thread.currentThread().isInterrupted()) {
                try {
                    Message msg = queue.poll(1L, TimeUnit.SECONDS);
                    if (msg != null) {
                        String json = gson.toJson(msg);
                        log.log(System.Logger.Level.DEBUG, "Publishing to '" + outgoingTopic + "': \n" + json);
                        publisher.sendMore(outgoingTopic);
                        publisher.send(json);
                    }
                }
                catch (InterruptedException e) {
                    log.log(System.Logger.Level.WARNING, "ZeroMQ Publisher thread was interrupted", e);
                    ok = false;
                }
                catch (Exception e) {
                    log.log(System.Logger.Level.WARNING, "An exception was thrown will attempting to publish a localization", e);
                }
            }
            log.log(System.Logger.Level.INFO, "Shutting down ZeroMQ publisher at " + address);
            publisher.close();
        });
        outgoingThread.setDaemon(true);
        outgoingThread.start();

        // subscriber
        incomingThread = new Thread(() -> {
            String address = "tcp://localhost:" + incomingPort;
            log.log(System.Logger.Level.INFO, "ZeroMQ Subscribing to " + address + " using topic '" + outgoingTopic + "'");
            Socket socket = context.createSocket(SocketType.SUB);
            socket.connect(address);
            socket.subscribe(incomingTopic.getBytes(ZMQ.CHARSET));
            while (ok && !Thread.currentThread().isInterrupted()) {
                try {
                    String topicAddress = socket.recvStr();
                    String contents = socket.recvStr();
                    log.log(System.Logger.Level.DEBUG, () -> "Received on '" + topicAddress + "':" + contents);
                    Message message = gson.fromJson(contents, Message.class);
                    controller.getIncoming().onNext(message);
                }
                catch (ZMQException e) {
                    if (e.getErrorCode() == 156384765) {
                        // do nothing
                    }
                    else {
                        log.log(System.Logger.Level.WARNING, "An exception occurred while reading from remote app", e);
                    }
                }
                catch (Exception e) {
                    log.log(System.Logger.Level.WARNING, "An exception occurred while reading from remote app", e);
                }
            }
        });
        incomingThread.setDaemon(true);
        incomingThread.start();
    }

    public IO(int incomingPort,
              int outgoingPort,
              String incomingTopic,
              String outgoingTopic) {
        this(incomingPort, outgoingPort, incomingTopic, outgoingTopic, new LocalizationController());
    }



    public int getIncomingPort() {
        return incomingPort;
    }

    public int getOutgoingPort() {
        return outgoingPort;
    }

    public LocalizationController getController() {
        return controller;
    }

    public SelectionController getSelectionController() {
        return selectionController;
    }

    public void publish(Message msg) {
        controller.getOutgoing().onNext(msg);
    }

    public void close() {
        ok = false;
        context.close();
        controller.getIncoming().onComplete();
        controller.getOutgoing().onComplete();
    }

    public Gson getGson() {
        return gson;
    }

    public String getSourceId() {
        return sourceId;
    }
}
