/*
 * Copyright 2014 the MechIO Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mechio.impl.speech;

import java.util.Properties;
import org.jflux.api.core.Listener;
import org.jflux.api.core.config.Configuration;
import org.jflux.api.messaging.rk.services.ServiceCommand;
import org.jflux.api.messaging.rk.services.ServiceError;
import org.jflux.impl.messaging.rk.config.DependentLifecycle;
import org.jflux.impl.messaging.rk.config.MessagingLifecycleGroupConfigUtils;
import org.jflux.impl.messaging.rk.config.RKMessagingConfigUtils;
import org.jflux.impl.services.rk.lifecycle.ManagedService;
import org.jflux.impl.services.rk.lifecycle.config.RKManagedGroupConfigUtils.ManagedGroupFactory;
import org.jflux.impl.services.rk.lifecycle.utils.ManagedServiceFactory;
import org.mechio.api.speech.SpeechConfig;
import org.mechio.api.speech.SpeechEventList;
import org.mechio.api.speech.SpeechRequest;
import org.mechio.api.speech.lifecycle.RemoteSpeechServiceClientLifecycle;
import org.mechio.api.speech.lifecycle.RemoteSpeechServiceHostLifecycle;
import org.mechio.api.speech.viseme.lifecycle.VisemeEventNotifierLifecycle;

import static org.jflux.impl.messaging.rk.config.MessagingLifecycleGroupConfigUtils.*;

/**
 *
 * @author Matthew Stevenson
 */
public class RemoteSpeechUtils {
    private final static String COMMAND_DEST_CONFIG_ID = "CommandDestConfig";
    private final static String CONFIG_DEST_CONFIG_ID = "ConfigDestConfig";
    private final static String ERROR_DEST_CONFIG_ID = "ErrorDestConfig";
    private final static String REQUEST_DEST_CONFIG_ID = "RequestDestConfig";
    private final static String EVENT_DEST_CONFIG_ID = "EventDestConfig";
    
    private final static String COMMAND_SERIALIZE_CONFIG_ID = ServiceCommand.class.toString();
    private final static String CONFIG_SERIALIZE_CONFIG_ID = SpeechConfig.class.toString();
    private final static String ERROR_SERIALIZE_CONFIG_ID = ServiceError.class.toString();
    private final static String REQUEST_SERIALIZE_CONFIG_ID = SpeechRequest.class.toString();
    private final static String EVENT_SERIALIZE_CONFIG_ID = SpeechEventList.class.toString();
    
    private final static String COMMAND_DEST_NAME = "Command";
    private final static String CONFIG_DEST_NAME = "Command";
    private final static String ERROR_DEST_NAME = "Error";
    private final static String REQUEST_DEST_NAME = "Request";
    private final static String EVENT_DEST_NAME = "Event";
    
    private final static String COMMAND_SENDER_ID = "Command";
    private final static String CONFIG_SENDER_ID = "Config";
    private final static String ERROR_RECEIVER_ID = "Error";
    private final static String REQUEST_SENDER_ID = "Request";
    private final static String EVENT_RECEIVER_ID = "Event";
    
    private final static String REQUEST_RECEIVER_ID = "RequestReceiver";
    private final static String EVENT_SENDER_ID = "EventSender";
        
    public final static String GROUP_PREFIX = "RKSpeechGroup";
    
    public static void connect(ManagedServiceFactory fact, 
            String speechGroupId, String speechPrefix, String connectionConfigId) {
        if(fact == null 
                || speechGroupId ==  null || connectionConfigId == null){
            throw new NullPointerException();
        }
        registerDestConfigs(speechGroupId, speechPrefix, fact);
        launchComponents(speechGroupId, connectionConfigId, speechPrefix, null, fact);
        
        fact.createService(
                new VisemeEventNotifierLifecycle(speechGroupId), null).start();
        launchRemoteSpeechClient(fact, speechGroupId, 
                speechGroupId,  
                speechPrefix + COMMAND_SENDER_ID, 
                speechPrefix + CONFIG_SENDER_ID, 
                speechPrefix + ERROR_RECEIVER_ID, 
                speechPrefix + REQUEST_SENDER_ID, 
                speechPrefix + EVENT_RECEIVER_ID);
    }
    
    public static void connectHost(
            ManagedServiceFactory fact, String speechGroupId,
            String speechPrefix, String connectionConfigId) {
        if(fact == null 
                || speechGroupId ==  null || connectionConfigId == null){
            throw new NullPointerException();
        }
        
        registerHostDestConfigs(speechGroupId, speechPrefix, fact);
        launchHostComponents(
                speechGroupId, connectionConfigId, speechPrefix, null, fact);
        
        launchRemoteSpeechHost(
                fact, speechGroupId, "", speechGroupId,
                speechPrefix + REQUEST_RECEIVER_ID,
                speechPrefix + EVENT_SENDER_ID);
    }
    
    private static void registerDestConfigs(String groupId, String speechPrefix, ManagedServiceFactory fact){
        String idBase =  groupId + "/" + GROUP_PREFIX;
        String destBase = ""; //groupId + GROUP_PREFIX;
        RKMessagingConfigUtils.registerQueueConfig(
                idBase + "/" + speechPrefix + COMMAND_DEST_CONFIG_ID, 
                destBase + speechPrefix + COMMAND_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerQueueConfig(
                idBase + "/" + speechPrefix + CONFIG_DEST_CONFIG_ID, 
                destBase + speechPrefix + CONFIG_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerTopicConfig(
                idBase + "/" + speechPrefix + ERROR_DEST_CONFIG_ID, 
                destBase + speechPrefix + ERROR_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerQueueConfig(
                idBase + "/" + speechPrefix + REQUEST_DEST_CONFIG_ID, 
                destBase + speechPrefix + REQUEST_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerTopicConfig(
                idBase + "/" + speechPrefix + EVENT_DEST_CONFIG_ID, 
                destBase + speechPrefix + EVENT_DEST_NAME,  null, fact);
    }
    
    private static void registerHostDestConfigs(
            String groupId, String speechPrefix, ManagedServiceFactory fact){
        String idBase =  groupId + "/" + GROUP_PREFIX;
        String destBase = ""; //groupId + GROUP_PREFIX;
        RKMessagingConfigUtils.registerQueueConfig(
                idBase + "/" + speechPrefix + REQUEST_DEST_CONFIG_ID, 
                destBase + speechPrefix + REQUEST_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerTopicConfig(
                idBase + "/" + speechPrefix + EVENT_DEST_CONFIG_ID, 
                destBase + speechPrefix + EVENT_DEST_NAME,  null, fact);
    }
    
    private static void launchComponents(
            String groupId, String connectionConfigId, String speechPrefix,
            Properties props, ManagedServiceFactory fact){
        String idBase = groupId + "/" + GROUP_PREFIX;
        launchComponent(
                idBase + "/" + speechPrefix + COMMAND_SENDER_ID, props, REMOTE_NOTIFIER, 
                idBase + "/" + speechPrefix + COMMAND_DEST_CONFIG_ID, connectionConfigId,
                COMMAND_SERIALIZE_CONFIG_ID, fact);
        launchComponent(
                idBase + "/" + speechPrefix + CONFIG_SENDER_ID, props, REMOTE_NOTIFIER, 
                idBase + "/" + speechPrefix + CONFIG_DEST_CONFIG_ID, connectionConfigId,
                CONFIG_SERIALIZE_CONFIG_ID, fact);
        launchComponent(
                idBase + "/" + speechPrefix + ERROR_RECEIVER_ID, props, REMOTE_LISTENER, 
                idBase + "/" + speechPrefix + ERROR_DEST_CONFIG_ID, connectionConfigId, 
                ERROR_SERIALIZE_CONFIG_ID, fact);
        launchComponent(
                idBase + "/" + speechPrefix + REQUEST_SENDER_ID, props, REMOTE_NOTIFIER, 
                idBase + "/" + speechPrefix + REQUEST_DEST_CONFIG_ID, connectionConfigId, 
                REQUEST_SERIALIZE_CONFIG_ID, fact);
        launchComponent(
                idBase + "/" + speechPrefix + EVENT_RECEIVER_ID, props, REMOTE_LISTENER, 
                idBase + "/" + speechPrefix + EVENT_DEST_CONFIG_ID, connectionConfigId, 
                EVENT_SERIALIZE_CONFIG_ID, fact);
    }
    
    private static void launchHostComponents(
            String groupId, String connectionConfigId, String speechPrefix,
            Properties props, ManagedServiceFactory fact){
        String idBase = groupId + "/" + GROUP_PREFIX;
        launchComponent(
                idBase + "/" + speechPrefix + REQUEST_RECEIVER_ID, props, REMOTE_LISTENER, 
                idBase + "/" + speechPrefix + REQUEST_DEST_CONFIG_ID, connectionConfigId, 
                REQUEST_SERIALIZE_CONFIG_ID, fact);
        launchComponent(
                idBase + "/" + speechPrefix + EVENT_SENDER_ID, props, REMOTE_NOTIFIER, 
                idBase + "/" + speechPrefix + EVENT_DEST_CONFIG_ID, connectionConfigId, 
                EVENT_SERIALIZE_CONFIG_ID, fact);
    }
    
    private static String launchComponent(
            final String groupId, 
            final Properties props, 
            final int componentType, 
            final String destinationConfigId, 
            final String connectionConfigId,
            String serializeConfigId, 
            ManagedServiceFactory fact){
        final ManagedGroupFactory mgf = new ManagedGroupFactory(fact); 
        DependentLifecycle.createDependencyListener(
                RKMessagingConfigUtils.SERIALIZATION_CONFIG, 
                serializeConfigId, Configuration.class, 
                new Listener<Configuration<String>>() {
                    @Override
                    public void handleEvent(Configuration<String> event) {
                        mgf.adapt(buildMessagingComponentLifecycleGroupConfig(
                                groupId, props, componentType, event, 
                                destinationConfigId, connectionConfigId));
                    }
                }, fact);
        return groupId(groupId, groupId, groupId);
    }
    
    private static void launchRemoteSpeechClient(
            ManagedServiceFactory fact,
            String speechClientId, String speechHostId,
            String commandSenderId, String configSenderId, 
            String errorReceiverId, String speechRequestSenderId,
            String speechEventsReceiverId){
        String idBase = speechClientId + "/" + GROUP_PREFIX;
        RemoteSpeechServiceClientLifecycle lifecycle =
                new RemoteSpeechServiceClientLifecycle(
                        speechClientId, speechHostId, 
                        groupId(idBase, commandSenderId, NOTIFIER_COMPONENT), 
                        groupId(idBase, configSenderId, NOTIFIER_COMPONENT), 
                        groupId(idBase, errorReceiverId, LISTENER_COMPONENT), 
                        groupId(idBase, speechRequestSenderId, NOTIFIER_COMPONENT), 
                        groupId(idBase, speechEventsReceiverId, LISTENER_COMPONENT));
        ManagedService speechComp = fact.createService(lifecycle, null);
        speechComp.start();
    }
    
    private static void launchRemoteSpeechHost(
            ManagedServiceFactory fact, String speechHostId, String remoteId,
            String speechServiceId, String requestReceiverId, 
            String speechEventSenderId){
        String idBase = speechHostId + "/" + GROUP_PREFIX;
        
        RemoteSpeechServiceHostLifecycle lifecycle =
                new RemoteSpeechServiceHostLifecycle(
                        speechHostId, remoteId, speechServiceId,
                        groupId(idBase, requestReceiverId, LISTENER_COMPONENT),
                        groupId(idBase, speechEventSenderId,
                                NOTIFIER_COMPONENT));
        ManagedService speechComp = fact.createService(lifecycle, null);
        speechComp.start();
    }
    
    private static String groupId(String groupId, String suffix, String component){
        return MessagingLifecycleGroupConfigUtils.childId(groupId + "/" + suffix, component);   
    }    
}
