/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org mybatis.io and other contributors.         ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.mapper.binding;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

import org.apache.ibatis.annotations.Lang;
import org.apache.ibatis.annotations.Param;
import org.apache.ibatis.annotations.UpdateProvider;
import org.miaixz.bus.mapper.Caching;
import org.miaixz.bus.mapper.binding.basic.EntityMapper;
import org.miaixz.bus.mapper.binding.condition.Condition;
import org.miaixz.bus.mapper.binding.condition.ConditionMapper;
import org.miaixz.bus.mapper.binding.condition.ConditionWrapper;
import org.miaixz.bus.mapper.binding.cursor.CursorMapper;
import org.miaixz.bus.mapper.binding.function.Fn;
import org.miaixz.bus.mapper.provider.FunctionProvider;

/**
 * 基础 Mapper 接口，集成常用操作方法，支持继承和方法覆盖
 *
 * @param <T> 实体类类型
 * @param <I> 主键类型
 * @author Kimi Liu
 * @since Java 17+
 */
public interface BasicMapper<T, I extends Serializable>
        extends EntityMapper<T, I>, ConditionMapper<T, Condition<T>>, CursorMapper<T, Condition<T>> {

    /**
     * 创建 Condition 查询封装对象
     *
     * @return ConditionWrapper 对象
     */
    default ConditionWrapper<T, I> wrapper() {
        return new ConditionWrapper<>(this, condition());
    }

    /**
     * 根据主键更新实体中非空字段，并强制更新指定字段
     *
     * @param entity 实体对象
     * @param fields 强制更新的字段集合，通过 {@link Fn#of(Fn...)} 创建 {@link Fn.FnArray}
     * @param <S>    实体类型
     * @return 1 表示成功，0 表示失败
     */
    @Lang(Caching.class)
    @UpdateProvider(type = FunctionProvider.class, method = "updateByPrimaryKeySelectiveWithForceFields")
    <S extends T> int updateByPrimaryKeySelectiveWithForceFields(@Param("entity") S entity,
            @Param("fns") Fn.FnArray<T> fields);

    /**
     * 根据主键更新指定字段集合
     *
     * @param entity 实体对象
     * @param fields 指定更新的字段集合，通过 {@link Fn#of(Fn...)} 创建 {@link Fn.FnArray}
     * @param <S>    实体类型
     * @return 1 表示成功，0 表示失败
     */
    @Lang(Caching.class)
    @UpdateProvider(type = FunctionProvider.class, method = "updateForFieldListByPrimaryKey")
    <S extends T> int updateForFieldListByPrimaryKey(@Param("entity") S entity, @Param("fns") Fn.FnArray<T> fields);

    /**
     * 根据指定字段集合查询，条件为 field IN (fieldValueList)
     *
     * @param field          字段方法引用
     * @param fieldValueList 字段值集合
     * @param <F>            字段类型
     * @return 符合条件的实体对象列表
     */
    default <F> List<T> selectByFieldList(Fn<T, F> field, Collection<F> fieldValueList) {
        Condition<T> condition = new Condition<>();
        condition.createCriteria().andIn((Fn<T, Object>) field.in(entityClass()), fieldValueList);
        return selectByCondition(condition);
    }

    /**
     * 根据指定字段集合删除，条件为 field IN (fieldValueList)
     *
     * @param field          字段方法引用
     * @param fieldValueList 字段值集合
     * @param <F>            字段类型
     * @return 删除的记录数
     */
    default <F> int deleteByFieldList(Fn<T, F> field, Collection<F> fieldValueList) {
        Condition<T> condition = new Condition<>();
        condition.createCriteria().andIn((Fn<T, Object>) field.in(entityClass()), fieldValueList);
        return deleteByCondition(condition);
    }

}