/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.media.server;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import org.apache.log4j.Logger;

import org.mobicents.media.server.impl.rtp.RTPManager;
import org.mobicents.media.server.io.network.UdpManager;
import org.mobicents.media.server.scheduler.Clock;
import org.mobicents.media.server.scheduler.Scheduler;
import org.mobicents.media.server.scheduler.Task;
import org.mobicents.media.server.spi.Endpoint;
import org.mobicents.media.server.spi.EndpointInstaller;
import org.mobicents.media.server.spi.MediaServer;
import org.mobicents.media.server.spi.ResourceUnavailableException;
import org.mobicents.media.server.spi.ServerManager;

/**
 *
 * @author Oifa Yulian
 */
public class Server implements MediaServer {

    //timing clock
    private Clock clock;

    //job scheduler
    private Scheduler scheduler;

    //The IP address to wich this server instance is bound
    private String bindAddress;

    //The UDP network manager
    private UdpManager udpManager;

    private NamingService namingService;
    private RTPManager rtpManager;

    //endpoint installers
    private ArrayList<EndpointInstaller> installers = new ArrayList();
    
    //endpoints
    private HashMap<String, Endpoint> endpoints = new HashMap();
    
    //managers
    private ArrayList<ServerManager> managers = new ArrayList();
    
    private HeartBeat heartbeat;
    private int heartbeatTime=0;
    private volatile long ttl;
    
    public  Logger logger = Logger.getLogger(Server.class);
    
    public Server() {
        namingService = new NamingService();
    }

   
    /**
     * Assigns clock instance.
     *
     * @param clock
     */
    public void setClock(Clock clock) {
        this.clock = clock;
    }

    /**
     * Get the address to which this server is bound
     *
     * @return
     */
    public String getBindAddress() {
        return bindAddress;
    }

    /**
     * Assigns the address to which this server will be bound.
     *
     * @param bindAddress the dotted IP address value
     */
    public void setBindAddress(String bindAddress) {
        this.bindAddress = bindAddress;
    }

    public void setScheduler(Scheduler scheduler) {
        this.scheduler = scheduler;
    }
    
    /**
     * Assigns the heartbeat time in minutes
     *
     * @param minutes
     */
    public void setHeartBeatTime(int heartbeatTime) {
        this.heartbeatTime = heartbeatTime;
    }

    /**
     * Gets the access to the RTP manager
     * @return
     */
    public RTPManager getRtpManager() {
        return rtpManager;
    }

    public void setInterface(UdpManager udpManager) {
        this.udpManager = udpManager;
    }
    
    /**
     * Installs endpoints defined by specified installer.
     *
     * @param installer the endpoints installer
     */
    public void addInstaller(EndpointInstaller installer) {
    	((VirtualEndpointInstaller)installer).setServer(this);
    	installers.add(installer);
        installer.install();        
    }

    /**
     * Uninstalls endpoint defined by specified endpoint installer.
     *
     * @param installer the endpoints installer.
     */
    public void removeInstaller(EndpointInstaller installer) {
        installers.remove(installer);
        installer.uninstall();
    }

    /**
     * Installs the specified endpoint.
     *
     * @param endpoint the endpoint to installed.
     */
    public void install(Endpoint endpoint) {
        //check endpoint first
        if (endpoint == null) {
            logger.error("Unknown endpoint");
            return;
        }

        //The endpoint implementation must extend BaseEndpointImpl class
        BaseEndpointImpl baseEndpoint = null;
        try {
            baseEndpoint = (BaseEndpointImpl) endpoint;
        } catch (ClassCastException e) {
            logger.error("Unsupported endpoint implementation " + endpoint.getLocalName());
            return;
        }


        //assign scheduler to the endpoint
        baseEndpoint.setScheduler(scheduler);
        baseEndpoint.setRtpManager(rtpManager);

        logger.info("Installing " + endpoint.getLocalName());

        //starting endpoint
        try {
            endpoint.start();
        } catch (Exception e) {
            logger.error("Couldn't start endpoint " + endpoint.getLocalName(), e);
            return;
        }

        //register endpoint with naming service
        try {
            namingService.register(endpoint);
        } catch (Exception e) {
            endpoint.stop();
            logger.error("Could not register endpoint " + endpoint.getLocalName(), e);
        }
        
        //register endpoint localy
        endpoints.put(endpoint.getLocalName(), endpoint);
        
        //send notification to manager
        for (ServerManager manager : managers) {
            manager.onStarted(endpoint);
        }
    }


    /**
     * Uninstalls the endpoint.
     *
     * @param name the local name of the endpoint to be uninstalled
     */
    public void uninstalls(String name) {
        //unregister localy
        Endpoint endpoint = endpoints.remove(name);

        //send notification to manager
        for (ServerManager manager : managers) {
            manager.onStarted(endpoint);
        }
        
        try {
            //TODO: lookup irrespective of endpoint usage
            endpoint = namingService.lookup(name, true);
            if (endpoint != null) {
                endpoint.stop();
                namingService.unregister(endpoint);
            }
        } catch (Exception e) {
        	logger.error(e);
        }
        
    }

    /**
     * Starts the server.
     *
     * @throws Exception
     */
    public void start() throws Exception {
        //check clock
        if (clock == null) {
            logger.error("Timing clock is not defined");
            return;
        }

        //check address configuration
        if (bindAddress == null) {
            logger.error("Bind address is not configured");
            return;
        }

        logger.info("Creating RTP Manager");
        rtpManager = new RTPManager(udpManager);
        rtpManager.setScheduler(scheduler);
        rtpManager.start();
        
        if(heartbeatTime>0)
        {
        	heartbeat=new HeartBeat(scheduler);
        	heartbeat.restart();
        }
    }

    /**
     * Stops the server.
     *
     */
    public void stop() {
        logger.info("Stopping UDP Manager");
        udpManager.stop();

        if(heartbeat!=null)
        	heartbeat.cancel();
        
        logger.info("Stopping scheduler");
        scheduler.stop();
        logger.info("Stopped media server instance ");                
    }

    public Endpoint lookup(String name, boolean bussy) throws ResourceUnavailableException {
        return null;//return namingService.lookup(name, bussy);
    }
    
    public Endpoint[] lookupall(String endpointName) throws ResourceUnavailableException {
    	return null;//return namingService.lookupall(endpointName);
    }

    public int getEndpointCount() {
        return 0;//return namingService.getEndpointCount();
    }
    
    public Collection<Endpoint> getEndpoints() {
        return endpoints.values();
    }

    /**
     * (Non Java-doc.)
     * 
     * @see org.mobicents.media.server.spi.MediaServer#addManager(org.mobicents.media.server.spi.ServerManager) 
     */
    public void addManager(ServerManager manager) {
        managers.add(manager);
    }

    /**
     * (Non Java-doc.)
     * 
     * @see org.mobicents.media.server.spi.MediaServer#removeManager(org.mobicents.media.server.spi.ServerManager) 
     */
    public void removeManager(ServerManager manager) {
        managers.remove(manager);
    }
    
    private class HeartBeat extends Task {

        public HeartBeat(Scheduler scheduler) {
            super(scheduler);
        }        

        public int getQueueNumber()
        {
        	return scheduler.HEARTBEAT_QUEUE;
        }   
        
        public void restart()
        {
        	ttl=heartbeatTime*600;
        	scheduler.submitHeatbeat(this);
        }
        
        @Override
        public long perform() {
        	ttl--;
            if (ttl == 0) {
            	logger.info("Global hearbeat is still alive");
            	restart();
            } else {
                scheduler.submitHeatbeat(this);
            }            
            return 0;
        }
    }
}
