package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.Project;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ProjectService;
import org.molgenis.omx.ngs.NgsUser;
import org.molgenis.controller.NgsUserController.NgsUserResponse;
import org.molgenis.service.NgsUserService;
import org.molgenis.omx.ngs.PrepKit;
import org.molgenis.controller.PrepKitController.PrepKitResponse;
import org.molgenis.service.PrepKitService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/project")
public class ProjectController
{
	@Autowired
	private ProjectService projectService;

	@Autowired
	private NgsUserService ngsUserService;
		
	@Autowired
	private PrepKitService prepKitService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ProjectResponse> createProject(@Valid @RequestBody ProjectRequest projectRequest)
			throws DatabaseException
	{
		return _createProject(projectRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProjectResponse> createProjectFromForm(@Valid @ModelAttribute ProjectRequest projectRequest)
			throws DatabaseException
	{
		return _createProject(projectRequest);
	}

	private ResponseEntity<ProjectResponse> _createProject(ProjectRequest projectRequest) throws DatabaseException
	{
		Project project = projectService.create(projectRequest.toProject());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/project/" + project.getId());
		return new ResponseEntity<ProjectResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ProjectResponse retrieveProject(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProject(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ProjectResponse retrieveProjectJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProject(id, expandFields);
	}

	private ProjectResponse _retrieveProject(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ProjectResponse(project, expandFields);
	}
			
	@RequestMapping(value = "/{id}/projectCustomer", method = RequestMethod.GET)
	public String retrieveProjectXrefProjectCustomer(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefProjectCustomer(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/projectCustomer", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProjectXrefProjectCustomerJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefProjectCustomer(id, "json", expandFields);
	}
	
	private String _retrieveProjectXrefProjectCustomer(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		Integer ngsUserId = project.getProjectCustomer_Id();
		String redirectUri = "redirect:/api/v1/ngsuser/" + ngsUserId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/prepKit", method = RequestMethod.GET)
	public String retrieveProjectXrefPrepKit(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefPrepKit(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/prepKit", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProjectXrefPrepKitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefPrepKit(id, "json", expandFields);
	}
	
	private String _retrieveProjectXrefPrepKit(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		Integer prepKitId = project.getPrepKit_Id();
		String redirectUri = "redirect:/api/v1/prepkit/" + prepKitId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/resultShippedUser", method = RequestMethod.GET)
	public String retrieveProjectXrefResultShippedUser(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefResultShippedUser(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/resultShippedUser", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProjectXrefResultShippedUserJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefResultShippedUser(id, "json", expandFields);
	}
	
	private String _retrieveProjectXrefResultShippedUser(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		Integer ngsUserId = project.getResultShippedUser_Id();
		String redirectUri = "redirect:/api/v1/ngsuser/" + ngsUserId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/resultShippedTo", method = RequestMethod.GET)
	public String retrieveProjectXrefResultShippedTo(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefResultShippedTo(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/resultShippedTo", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProjectXrefResultShippedToJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectXrefResultShippedTo(id, "json", expandFields);
	}
	
	private String _retrieveProjectXrefResultShippedTo(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		Integer ngsUserId = project.getResultShippedTo_Id();
		String redirectUri = "redirect:/api/v1/ngsuser/" + ngsUserId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/projectAnalist", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<NgsUserResponse> retrieveProjectMrefProjectAnalist(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		return _retrieveProjectMrefProjectAnalist(project, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/projectAnalist", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<NgsUserResponse> retrieveProjectMrefProjectAnalistJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Project project = projectService.read(id);
		if (project == null) throw new EntityNotFoundException("Project " + id.toString() + " not found");
		return _retrieveProjectMrefProjectAnalist(project, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<NgsUserResponse> _retrieveProjectMrefProjectAnalist(Project project, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<NgsUser> ngsUserCollection = project.getProjectAnalist();
		
		int total = ngsUserCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		ngsUserCollection = ngsUserCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<NgsUser> ngsUserPager = new EntityPager<NgsUser>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, ngsUserCollection);
		
		return new EntityCollectionResponse<NgsUserResponse>(ngsUserPager, Lists.newArrayList(Iterables.transform(ngsUserCollection,
				new Function<NgsUser, NgsUserResponse>()
				{
					@Override
					@Nullable
					public NgsUserResponse apply(@Nullable NgsUser ngsUser)
					{
						try
						{
							return ngsUser != null ? new NgsUserResponse(ngsUser, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/project/" + project.getId() + "/projectAnalist");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateProject(@PathVariable Integer id, @Valid @RequestBody ProjectRequest projectRequest)
			throws DatabaseException
	{
		_updateProject(id, projectRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProjectResponse> updateProjectFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ProjectRequest projectRequest) throws DatabaseException
	{
		return _createProject(projectRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProjectPost(@PathVariable Integer id, @Valid @RequestBody ProjectRequest projectRequest)
			throws DatabaseException
	{
		_updateProject(id, projectRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProjectFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ProjectRequest projectRequest)
			throws DatabaseException
	{
		_updateProject(id, projectRequest);
	}

	private void _updateProject(Integer id, ProjectRequest projectRequest) throws DatabaseException
	{
		Project project = projectRequest.toProject();
		project.setId(id);
		projectService.update(project);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProject(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProject(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProjectPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProject(id);
	}

	private void _deleteProject(Integer id) throws DatabaseException
	{
		boolean isDeleted = projectService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Project " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProjectResponse> retrieveProjectCollection(@Valid EntityCollectionRequest projectCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectCollection(projectCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProjectResponse> retrieveProjectCollectionJson(@Valid EntityCollectionRequest projectCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectCollection(projectCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ProjectResponse> retrieveProjectCollectionPost(@Valid @RequestBody EntityCollectionRequest projectCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectCollection(projectCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProjectResponse> retrieveProjectCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest projectCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProjectCollection(projectCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ProjectResponse> _retrieveProjectCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Project> projectPager = projectService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ProjectResponse>(projectPager, Lists.newArrayList(Iterables.transform(
				projectPager.getIterable(), new Function<Project, ProjectResponse>()
				{
					@Override
					@Nullable
					public ProjectResponse apply(@Nullable Project project)
					{
						try
						{
							return project != null ? new ProjectResponse(project, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/project");
	}

	private static class ProjectRequest
	{
		private String projectName;
		private String projectComment;
		private Integer projectCustomer;
		private java.util.List<Integer> projectAnalist;
		private java.util.Date projectPlannedFinishDate;
		private Integer laneAmount;
		private Integer sampleAmount;
		private String sampleType;
		private String seqType;
		private Integer prepKit;
		private String declarationNr;
		private Boolean gccAnalysis;
		private String gccInstructions;
		private String resultFilesDir;
		private Integer resultShippedUser;
		private Integer resultShippedTo;
		private java.util.Date resultShippedDate;
	
		public Project toProject()
		{
			Project project = new Project();
			project.setProjectName(projectName);
			project.setProjectComment(projectComment);
			project.setProjectCustomer_Id(projectCustomer);
			project.setProjectAnalist_Id(projectAnalist);
			project.setProjectPlannedFinishDate(projectPlannedFinishDate);
			project.setLaneAmount(laneAmount);
			project.setSampleAmount(sampleAmount);
			project.setSampleType(sampleType);
			project.setSeqType(seqType);
			project.setPrepKit_Id(prepKit);
			project.setDeclarationNr(declarationNr);
			project.setGccAnalysis(gccAnalysis);
			project.setGccInstructions(gccInstructions);
			project.setResultFilesDir(resultFilesDir);
			project.setResultShippedUser_Id(resultShippedUser);
			project.setResultShippedTo_Id(resultShippedTo);
			project.setResultShippedDate(resultShippedDate);
			return project;
		}
		
		public void setProjectName(String projectName)
		{
			this.projectName = projectName;
		}
		
		public void setProjectComment(String projectComment)
		{
			this.projectComment = projectComment;
		}
		
		public void setProjectCustomer(Integer projectCustomer)
		{
			this.projectCustomer = projectCustomer;
		}
		
		public void setProjectAnalist(java.util.List<Integer> projectAnalist)
		{
			this.projectAnalist = projectAnalist;
		}
		
		public void setProjectPlannedFinishDate(java.util.Date projectPlannedFinishDate)
		{
			this.projectPlannedFinishDate = projectPlannedFinishDate;
		}
		
		public void setLaneAmount(Integer laneAmount)
		{
			this.laneAmount = laneAmount;
		}
		
		public void setSampleAmount(Integer sampleAmount)
		{
			this.sampleAmount = sampleAmount;
		}
		
		public void setSampleType(String sampleType)
		{
			this.sampleType = sampleType;
		}
		
		public void setSeqType(String seqType)
		{
			this.seqType = seqType;
		}
		
		public void setPrepKit(Integer prepKit)
		{
			this.prepKit = prepKit;
		}
		
		public void setDeclarationNr(String declarationNr)
		{
			this.declarationNr = declarationNr;
		}
		
		public void setGccAnalysis(Boolean gccAnalysis)
		{
			this.gccAnalysis = gccAnalysis;
		}
		
		public void setGccInstructions(String gccInstructions)
		{
			this.gccInstructions = gccInstructions;
		}
		
		public void setResultFilesDir(String resultFilesDir)
		{
			this.resultFilesDir = resultFilesDir;
		}
		
		public void setResultShippedUser(Integer resultShippedUser)
		{
			this.resultShippedUser = resultShippedUser;
		}
		
		public void setResultShippedTo(Integer resultShippedTo)
		{
			this.resultShippedTo = resultShippedTo;
		}
		
		public void setResultShippedDate(java.util.Date resultShippedDate)
		{
			this.resultShippedDate = resultShippedDate;
		}
		
	}

	static class ProjectResponse
	{
		private final String href;
		private final String projectName;
		private final String projectComment;
		private final Object projectCustomer;
		private final Object projectAnalist;
		private final java.util.Date projectPlannedFinishDate;
		private final Integer laneAmount;
		private final Integer sampleAmount;
		private final String sampleType;
		private final String seqType;
		private final Object prepKit;
		private final String declarationNr;
		private final Boolean gccAnalysis;
		private final String gccInstructions;
		private final String resultFilesDir;
		private final Object resultShippedUser;
		private final Object resultShippedTo;
		private final java.util.Date resultShippedDate;
	
		public ProjectResponse(Project project, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/project/" + project.getId();
			this.projectName = project.getProjectName();
			this.projectComment = project.getProjectComment();
			if (expandFields != null && expandFields.contains("projectCustomer")) this.projectCustomer = project.getProjectCustomer() == null ? null : new NgsUserResponse(project.getProjectCustomer(), null);
			else this.projectCustomer = project.getProjectCustomer() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/project/" + project.getId() + "/projectCustomer");
			java.util.List<NgsUser> ProjectAnalistCollection = project.getProjectAnalist();
			if (expandFields != null && expandFields.contains("projectAnalist")) this.projectAnalist = ProjectAnalistCollection == null ? null : _retrieveProjectMrefProjectAnalist(project, new EntityCollectionRequest());
			else this.projectAnalist = ProjectAnalistCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/project/" + project.getId() + "/projectAnalist");
			this.projectPlannedFinishDate = project.getProjectPlannedFinishDate();
			this.laneAmount = project.getLaneAmount();
			this.sampleAmount = project.getSampleAmount();
			this.sampleType = project.getSampleType();
			this.seqType = project.getSeqType();
			if (expandFields != null && expandFields.contains("prepKit")) this.prepKit = project.getPrepKit() == null ? null : new PrepKitResponse(project.getPrepKit(), null);
			else this.prepKit = project.getPrepKit() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/project/" + project.getId() + "/prepKit");
			this.declarationNr = project.getDeclarationNr();
			this.gccAnalysis = project.getGccAnalysis();
			this.gccInstructions = project.getGccInstructions();
			this.resultFilesDir = project.getResultFilesDir();
			if (expandFields != null && expandFields.contains("resultShippedUser")) this.resultShippedUser = project.getResultShippedUser() == null ? null : new NgsUserResponse(project.getResultShippedUser(), null);
			else this.resultShippedUser = project.getResultShippedUser() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/project/" + project.getId() + "/resultShippedUser");
			if (expandFields != null && expandFields.contains("resultShippedTo")) this.resultShippedTo = project.getResultShippedTo() == null ? null : new NgsUserResponse(project.getResultShippedTo(), null);
			else this.resultShippedTo = project.getResultShippedTo() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/project/" + project.getId() + "/resultShippedTo");
			this.resultShippedDate = project.getResultShippedDate();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getProjectName()
		{
			return projectName;
		}
	
		public String getProjectComment()
		{
			return projectComment;
		}
	
		public Object getProjectCustomer()
		{
			return projectCustomer;
		}
	
		public Object getProjectAnalist()
		{
			return projectAnalist;
		}
	
		public java.util.Date getProjectPlannedFinishDate()
		{
			return projectPlannedFinishDate;
		}
	
		public Integer getLaneAmount()
		{
			return laneAmount;
		}
	
		public Integer getSampleAmount()
		{
			return sampleAmount;
		}
	
		public String getSampleType()
		{
			return sampleType;
		}
	
		public String getSeqType()
		{
			return seqType;
		}
	
		public Object getPrepKit()
		{
			return prepKit;
		}
	
		public String getDeclarationNr()
		{
			return declarationNr;
		}
	
		public Boolean getGccAnalysis()
		{
			return gccAnalysis;
		}
	
		public String getGccInstructions()
		{
			return gccInstructions;
		}
	
		public String getResultFilesDir()
		{
			return resultFilesDir;
		}
	
		public Object getResultShippedUser()
		{
			return resultShippedUser;
		}
	
		public Object getResultShippedTo()
		{
			return resultShippedTo;
		}
	
		public java.util.Date getResultShippedDate()
		{
			return resultShippedDate;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}