package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.XrefValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.XrefValueService;
import org.molgenis.omx.observ.Characteristic;
import org.molgenis.controller.CharacteristicController.CharacteristicResponse;
import org.molgenis.service.CharacteristicService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/xrefvalue")
public class XrefValueController
{
	@Autowired
	private XrefValueService xrefValueService;

	@Autowired
	private CharacteristicService characteristicService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<XrefValueResponse> createXrefValue(@Valid @RequestBody XrefValueRequest xrefValueRequest)
			throws DatabaseException
	{
		return _createXrefValue(xrefValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<XrefValueResponse> createXrefValueFromForm(@Valid @ModelAttribute XrefValueRequest xrefValueRequest)
			throws DatabaseException
	{
		return _createXrefValue(xrefValueRequest);
	}

	private ResponseEntity<XrefValueResponse> _createXrefValue(XrefValueRequest xrefValueRequest) throws DatabaseException
	{
		XrefValue xrefValue = xrefValueService.create(xrefValueRequest.toXrefValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/xrefvalue/" + xrefValue.getId());
		return new ResponseEntity<XrefValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public XrefValueResponse retrieveXrefValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public XrefValueResponse retrieveXrefValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValue(id, expandFields);
	}

	private XrefValueResponse _retrieveXrefValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		XrefValue xrefValue = xrefValueService.read(id);
		if (xrefValue == null) throw new EntityNotFoundException("XrefValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new XrefValueResponse(xrefValue, expandFields);
	}
			
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET)
	public String retrieveXrefValueXrefValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValueXrefValue(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveXrefValueXrefValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValueXrefValue(id, "json", expandFields);
	}
	
	private String _retrieveXrefValueXrefValue(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		XrefValue xrefValue = xrefValueService.read(id);
		if (xrefValue == null) throw new EntityNotFoundException("XrefValue " + id.toString() + " not found");
		Integer characteristicId = xrefValue.getValue_Id();
		String redirectUri = "redirect:/api/v1/characteristic/" + characteristicId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateXrefValue(@PathVariable Integer id, @Valid @RequestBody XrefValueRequest xrefValueRequest)
			throws DatabaseException
	{
		_updateXrefValue(id, xrefValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<XrefValueResponse> updateXrefValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute XrefValueRequest xrefValueRequest) throws DatabaseException
	{
		return _createXrefValue(xrefValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateXrefValuePost(@PathVariable Integer id, @Valid @RequestBody XrefValueRequest xrefValueRequest)
			throws DatabaseException
	{
		_updateXrefValue(id, xrefValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateXrefValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute XrefValueRequest xrefValueRequest)
			throws DatabaseException
	{
		_updateXrefValue(id, xrefValueRequest);
	}

	private void _updateXrefValue(Integer id, XrefValueRequest xrefValueRequest) throws DatabaseException
	{
		XrefValue xrefValue = xrefValueRequest.toXrefValue();
		xrefValue.setId(id);
		xrefValueService.update(xrefValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteXrefValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteXrefValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteXrefValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteXrefValue(id);
	}

	private void _deleteXrefValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = xrefValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("XrefValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<XrefValueResponse> retrieveXrefValueCollection(@Valid EntityCollectionRequest xrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValueCollection(xrefValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<XrefValueResponse> retrieveXrefValueCollectionJson(@Valid EntityCollectionRequest xrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValueCollection(xrefValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<XrefValueResponse> retrieveXrefValueCollectionPost(@Valid @RequestBody EntityCollectionRequest xrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValueCollection(xrefValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<XrefValueResponse> retrieveXrefValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest xrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveXrefValueCollection(xrefValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<XrefValueResponse> _retrieveXrefValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<XrefValue> xrefValuePager = xrefValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<XrefValueResponse>(xrefValuePager, Lists.newArrayList(Iterables.transform(
				xrefValuePager.getIterable(), new Function<XrefValue, XrefValueResponse>()
				{
					@Override
					@Nullable
					public XrefValueResponse apply(@Nullable XrefValue xrefValue)
					{
						try
						{
							return xrefValue != null ? new XrefValueResponse(xrefValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/xrefvalue");
	}

	private static class XrefValueRequest
	{
		private Integer value;
	
		public XrefValue toXrefValue()
		{
			XrefValue xrefValue = new XrefValue();
			xrefValue.setValue_Id(value);
			return xrefValue;
		}
		
		public void setValue(Integer value)
		{
			this.value = value;
		}
		
	}

	static class XrefValueResponse
	{
		private final String href;
		private final Object value;
	
		public XrefValueResponse(XrefValue xrefValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/xrefvalue/" + xrefValue.getId();
			if (expandFields != null && expandFields.contains("value")) this.value = new CharacteristicResponse(xrefValue.getValue(), null);
			else this.value = java.util.Collections.singletonMap("href", "/api/v1/xrefvalue/" + xrefValue.getId() + "/value");
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public Object getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}