package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.target.Accession;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.AccessionService;
import org.molgenis.omx.observ.target.Ontology;
import org.molgenis.controller.OntologyController.OntologyResponse;
import org.molgenis.service.OntologyService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/accession")
public class AccessionController
{
	@Autowired
	private AccessionService accessionService;

	@Autowired
	private OntologyService ontologyService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<AccessionResponse> createAccession(@Valid @RequestBody AccessionRequest accessionRequest)
			throws DatabaseException
	{
		return _createAccession(accessionRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<AccessionResponse> createAccessionFromForm(@Valid @ModelAttribute AccessionRequest accessionRequest)
			throws DatabaseException
	{
		return _createAccession(accessionRequest);
	}

	private ResponseEntity<AccessionResponse> _createAccession(AccessionRequest accessionRequest) throws DatabaseException
	{
		Accession accession = accessionService.create(accessionRequest.toAccession());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/accession/" + accession.getId());
		return new ResponseEntity<AccessionResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public AccessionResponse retrieveAccession(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccession(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public AccessionResponse retrieveAccessionJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccession(id, expandFields);
	}

	private AccessionResponse _retrieveAccession(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Accession accession = accessionService.read(id);
		if (accession == null) throw new EntityNotFoundException("Accession " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new AccessionResponse(accession, expandFields);
	}
			
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET)
	public String retrieveAccessionXrefOntology(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccessionXrefOntology(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveAccessionXrefOntologyJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccessionXrefOntology(id, "json", expandFields);
	}
	
	private String _retrieveAccessionXrefOntology(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Accession accession = accessionService.read(id);
		if (accession == null) throw new EntityNotFoundException("Accession " + id.toString() + " not found");
		Integer ontologyId = accession.getOntology_Id();
		String redirectUri = "redirect:/api/v1/ontology/" + ontologyId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateAccession(@PathVariable Integer id, @Valid @RequestBody AccessionRequest accessionRequest)
			throws DatabaseException
	{
		_updateAccession(id, accessionRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<AccessionResponse> updateAccessionFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute AccessionRequest accessionRequest) throws DatabaseException
	{
		return _createAccession(accessionRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateAccessionPost(@PathVariable Integer id, @Valid @RequestBody AccessionRequest accessionRequest)
			throws DatabaseException
	{
		_updateAccession(id, accessionRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateAccessionFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute AccessionRequest accessionRequest)
			throws DatabaseException
	{
		_updateAccession(id, accessionRequest);
	}

	private void _updateAccession(Integer id, AccessionRequest accessionRequest) throws DatabaseException
	{
		Accession accession = accessionRequest.toAccession();
		accession.setId(id);
		accessionService.update(accession);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteAccession(@PathVariable Integer id) throws DatabaseException
	{
		_deleteAccession(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteAccessionPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteAccession(id);
	}

	private void _deleteAccession(Integer id) throws DatabaseException
	{
		boolean isDeleted = accessionService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Accession " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<AccessionResponse> retrieveAccessionCollection(@Valid EntityCollectionRequest accessionCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccessionCollection(accessionCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<AccessionResponse> retrieveAccessionCollectionJson(@Valid EntityCollectionRequest accessionCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccessionCollection(accessionCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<AccessionResponse> retrieveAccessionCollectionPost(@Valid @RequestBody EntityCollectionRequest accessionCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccessionCollection(accessionCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<AccessionResponse> retrieveAccessionCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest accessionCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveAccessionCollection(accessionCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<AccessionResponse> _retrieveAccessionCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Accession> accessionPager = accessionService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<AccessionResponse>(accessionPager, Lists.newArrayList(Iterables.transform(
				accessionPager.getIterable(), new Function<Accession, AccessionResponse>()
				{
					@Override
					@Nullable
					public AccessionResponse apply(@Nullable Accession accession)
					{
						try
						{
							return accession != null ? new AccessionResponse(accession, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/accession");
	}

	private static class AccessionRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer ontology;
		private String termAccession;
		private String definition;
	
		public Accession toAccession()
		{
			Accession accession = new Accession();
			accession.setIdentifier(identifier);
			accession.setName(name);
			accession.setDescription(description);
			accession.setOntology_Id(ontology);
			accession.setTermAccession(termAccession);
			accession.setDefinition(definition);
			return accession;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setOntology(Integer ontology)
		{
			this.ontology = ontology;
		}
		
		public void setTermAccession(String termAccession)
		{
			this.termAccession = termAccession;
		}
		
		public void setDefinition(String definition)
		{
			this.definition = definition;
		}
		
	}

	static class AccessionResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object ontology;
		private final String termAccession;
		private final String definition;
	
		public AccessionResponse(Accession accession, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/accession/" + accession.getId();
			this.identifier = accession.getIdentifier();
			this.name = accession.getName();
			this.description = accession.getDescription();
			if (expandFields != null && expandFields.contains("ontology")) this.ontology = accession.getOntology() == null ? null : new OntologyResponse(accession.getOntology(), null);
			else this.ontology = accession.getOntology() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/accession/" + accession.getId() + "/ontology");
			this.termAccession = accession.getTermAccession();
			this.definition = accession.getDefinition();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getOntology()
		{
			return ontology;
		}
	
		public String getTermAccession()
		{
			return termAccession;
		}
	
		public String getDefinition()
		{
			return definition;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}