package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.EmailValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.EmailValueService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/emailvalue")
public class EmailValueController
{
	@Autowired
	private EmailValueService emailValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<EmailValueResponse> createEmailValue(@Valid @RequestBody EmailValueRequest emailValueRequest)
			throws DatabaseException
	{
		return _createEmailValue(emailValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<EmailValueResponse> createEmailValueFromForm(@Valid @ModelAttribute EmailValueRequest emailValueRequest)
			throws DatabaseException
	{
		return _createEmailValue(emailValueRequest);
	}

	private ResponseEntity<EmailValueResponse> _createEmailValue(EmailValueRequest emailValueRequest) throws DatabaseException
	{
		EmailValue emailValue = emailValueService.create(emailValueRequest.toEmailValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/emailvalue/" + emailValue.getId());
		return new ResponseEntity<EmailValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public EmailValueResponse retrieveEmailValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEmailValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EmailValueResponse retrieveEmailValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEmailValue(id, expandFields);
	}

	private EmailValueResponse _retrieveEmailValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		EmailValue emailValue = emailValueService.read(id);
		if (emailValue == null) throw new EntityNotFoundException("EmailValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EmailValueResponse(emailValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateEmailValue(@PathVariable Integer id, @Valid @RequestBody EmailValueRequest emailValueRequest)
			throws DatabaseException
	{
		_updateEmailValue(id, emailValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<EmailValueResponse> updateEmailValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute EmailValueRequest emailValueRequest) throws DatabaseException
	{
		return _createEmailValue(emailValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateEmailValuePost(@PathVariable Integer id, @Valid @RequestBody EmailValueRequest emailValueRequest)
			throws DatabaseException
	{
		_updateEmailValue(id, emailValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateEmailValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute EmailValueRequest emailValueRequest)
			throws DatabaseException
	{
		_updateEmailValue(id, emailValueRequest);
	}

	private void _updateEmailValue(Integer id, EmailValueRequest emailValueRequest) throws DatabaseException
	{
		EmailValue emailValue = emailValueRequest.toEmailValue();
		emailValue.setId(id);
		emailValueService.update(emailValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteEmailValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteEmailValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteEmailValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteEmailValue(id);
	}

	private void _deleteEmailValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = emailValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("EmailValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<EmailValueResponse> retrieveEmailValueCollection(@Valid EntityCollectionRequest emailValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEmailValueCollection(emailValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<EmailValueResponse> retrieveEmailValueCollectionJson(@Valid EntityCollectionRequest emailValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEmailValueCollection(emailValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<EmailValueResponse> retrieveEmailValueCollectionPost(@Valid @RequestBody EntityCollectionRequest emailValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEmailValueCollection(emailValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<EmailValueResponse> retrieveEmailValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest emailValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEmailValueCollection(emailValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<EmailValueResponse> _retrieveEmailValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<EmailValue> emailValuePager = emailValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<EmailValueResponse>(emailValuePager, Lists.newArrayList(Iterables.transform(
				emailValuePager.getIterable(), new Function<EmailValue, EmailValueResponse>()
				{
					@Override
					@Nullable
					public EmailValueResponse apply(@Nullable EmailValue emailValue)
					{
						try
						{
							return emailValue != null ? new EmailValueResponse(emailValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/emailvalue");
	}

	private static class EmailValueRequest
	{
		private String value;
	
		public EmailValue toEmailValue()
		{
			EmailValue emailValue = new EmailValue();
			emailValue.setValue(value);
			return emailValue;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class EmailValueResponse
	{
		private final String href;
		private final String value;
	
		public EmailValueResponse(EmailValue emailValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/emailvalue/" + emailValue.getId();
			this.value = emailValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}