package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/observablefeature")
public class ObservableFeatureController
{
	@Autowired
	private ObservableFeatureService observableFeatureService;

	@Autowired
	private OntologyTermService ontologyTermService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ObservableFeatureResponse> createObservableFeature(@Valid @RequestBody ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		return _createObservableFeature(observableFeatureRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservableFeatureResponse> createObservableFeatureFromForm(@Valid @ModelAttribute ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		return _createObservableFeature(observableFeatureRequest);
	}

	private ResponseEntity<ObservableFeatureResponse> _createObservableFeature(ObservableFeatureRequest observableFeatureRequest) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.create(observableFeatureRequest.toObservableFeature());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/observablefeature/" + observableFeature.getId());
		return new ResponseEntity<ObservableFeatureResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ObservableFeatureResponse retrieveObservableFeature(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeature(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ObservableFeatureResponse retrieveObservableFeatureJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeature(id, expandFields);
	}

	private ObservableFeatureResponse _retrieveObservableFeature(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ObservableFeatureResponse(observableFeature, expandFields);
	}
			
	@RequestMapping(value = "/{id}/unit", method = RequestMethod.GET)
	public String retrieveObservableFeatureXrefUnit(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureXrefUnit(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/unit", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservableFeatureXrefUnitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureXrefUnit(id, "json", expandFields);
	}
	
	private String _retrieveObservableFeatureXrefUnit(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		Integer ontologyTermId = observableFeature.getUnit_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/definition", method = RequestMethod.GET)
	public String retrieveObservableFeatureXrefDefinition(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureXrefDefinition(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/definition", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservableFeatureXrefDefinitionJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureXrefDefinition(id, "json", expandFields);
	}
	
	private String _retrieveObservableFeatureXrefDefinition(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		Integer ontologyTermId = observableFeature.getDefinition_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateObservableFeature(@PathVariable Integer id, @Valid @RequestBody ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		_updateObservableFeature(id, observableFeatureRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservableFeatureResponse> updateObservableFeatureFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ObservableFeatureRequest observableFeatureRequest) throws DatabaseException
	{
		return _createObservableFeature(observableFeatureRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservableFeaturePost(@PathVariable Integer id, @Valid @RequestBody ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		_updateObservableFeature(id, observableFeatureRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservableFeatureFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		_updateObservableFeature(id, observableFeatureRequest);
	}

	private void _updateObservableFeature(Integer id, ObservableFeatureRequest observableFeatureRequest) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureRequest.toObservableFeature();
		observableFeature.setId(id);
		observableFeatureService.update(observableFeature);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservableFeature(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservableFeature(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservableFeaturePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservableFeature(id);
	}

	private void _deleteObservableFeature(Integer id) throws DatabaseException
	{
		boolean isDeleted = observableFeatureService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollection(@Valid EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollectionJson(@Valid EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollectionPost(@Valid @RequestBody EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ObservableFeatureResponse> _retrieveObservableFeatureCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ObservableFeature> observableFeaturePager = observableFeatureService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ObservableFeatureResponse>(observableFeaturePager, Lists.newArrayList(Iterables.transform(
				observableFeaturePager.getIterable(), new Function<ObservableFeature, ObservableFeatureResponse>()
				{
					@Override
					@Nullable
					public ObservableFeatureResponse apply(@Nullable ObservableFeature observableFeature)
					{
						try
						{
							return observableFeature != null ? new ObservableFeatureResponse(observableFeature, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/observablefeature");
	}

	private static class ObservableFeatureRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer unit;
		private Integer definition;
		private String dataType;
		private Boolean temporal;
	
		public ObservableFeature toObservableFeature()
		{
			ObservableFeature observableFeature = new ObservableFeature();
			observableFeature.setIdentifier(identifier);
			observableFeature.setName(name);
			observableFeature.setDescription(description);
			observableFeature.setUnit_Id(unit);
			observableFeature.setDefinition_Id(definition);
			observableFeature.setDataType(dataType);
			observableFeature.setTemporal(temporal);
			return observableFeature;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setUnit(Integer unit)
		{
			this.unit = unit;
		}
		
		public void setDefinition(Integer definition)
		{
			this.definition = definition;
		}
		
		public void setDataType(String dataType)
		{
			this.dataType = dataType;
		}
		
		public void setTemporal(Boolean temporal)
		{
			this.temporal = temporal;
		}
		
	}

	static class ObservableFeatureResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object unit;
		private final Object definition;
		private final String dataType;
		private final Boolean temporal;
	
		public ObservableFeatureResponse(ObservableFeature observableFeature, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/observablefeature/" + observableFeature.getId();
			this.identifier = observableFeature.getIdentifier();
			this.name = observableFeature.getName();
			this.description = observableFeature.getDescription();
			if (expandFields != null && expandFields.contains("unit")) this.unit = observableFeature.getUnit() == null ? null : new OntologyTermResponse(observableFeature.getUnit(), null);
			else this.unit = observableFeature.getUnit() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/observablefeature/" + observableFeature.getId() + "/unit");
			if (expandFields != null && expandFields.contains("definition")) this.definition = observableFeature.getDefinition() == null ? null : new OntologyTermResponse(observableFeature.getDefinition(), null);
			else this.definition = observableFeature.getDefinition() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/observablefeature/" + observableFeature.getId() + "/definition");
			this.dataType = observableFeature.getDataType();
			this.temporal = observableFeature.getTemporal();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getUnit()
		{
			return unit;
		}
	
		public Object getDefinition()
		{
			return definition;
		}
	
		public String getDataType()
		{
			return dataType;
		}
	
		public Boolean getTemporal()
		{
			return temporal;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}