package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.target.Ontology;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.OntologyService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/ontology")
public class OntologyController
{
	@Autowired
	private OntologyService ontologyService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<OntologyResponse> createOntology(@Valid @RequestBody OntologyRequest ontologyRequest)
			throws DatabaseException
	{
		return _createOntology(ontologyRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<OntologyResponse> createOntologyFromForm(@Valid @ModelAttribute OntologyRequest ontologyRequest)
			throws DatabaseException
	{
		return _createOntology(ontologyRequest);
	}

	private ResponseEntity<OntologyResponse> _createOntology(OntologyRequest ontologyRequest) throws DatabaseException
	{
		Ontology ontology = ontologyService.create(ontologyRequest.toOntology());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/ontology/" + ontology.getId());
		return new ResponseEntity<OntologyResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public OntologyResponse retrieveOntology(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntology(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public OntologyResponse retrieveOntologyJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntology(id, expandFields);
	}

	private OntologyResponse _retrieveOntology(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Ontology ontology = ontologyService.read(id);
		if (ontology == null) throw new EntityNotFoundException("Ontology " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new OntologyResponse(ontology, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateOntology(@PathVariable Integer id, @Valid @RequestBody OntologyRequest ontologyRequest)
			throws DatabaseException
	{
		_updateOntology(id, ontologyRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<OntologyResponse> updateOntologyFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute OntologyRequest ontologyRequest) throws DatabaseException
	{
		return _createOntology(ontologyRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateOntologyPost(@PathVariable Integer id, @Valid @RequestBody OntologyRequest ontologyRequest)
			throws DatabaseException
	{
		_updateOntology(id, ontologyRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateOntologyFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute OntologyRequest ontologyRequest)
			throws DatabaseException
	{
		_updateOntology(id, ontologyRequest);
	}

	private void _updateOntology(Integer id, OntologyRequest ontologyRequest) throws DatabaseException
	{
		Ontology ontology = ontologyRequest.toOntology();
		ontology.setId(id);
		ontologyService.update(ontology);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteOntology(@PathVariable Integer id) throws DatabaseException
	{
		_deleteOntology(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteOntologyPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteOntology(id);
	}

	private void _deleteOntology(Integer id) throws DatabaseException
	{
		boolean isDeleted = ontologyService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Ontology " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<OntologyResponse> retrieveOntologyCollection(@Valid EntityCollectionRequest ontologyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyCollection(ontologyCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<OntologyResponse> retrieveOntologyCollectionJson(@Valid EntityCollectionRequest ontologyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyCollection(ontologyCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<OntologyResponse> retrieveOntologyCollectionPost(@Valid @RequestBody EntityCollectionRequest ontologyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyCollection(ontologyCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<OntologyResponse> retrieveOntologyCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest ontologyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyCollection(ontologyCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<OntologyResponse> _retrieveOntologyCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Ontology> ontologyPager = ontologyService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<OntologyResponse>(ontologyPager, Lists.newArrayList(Iterables.transform(
				ontologyPager.getIterable(), new Function<Ontology, OntologyResponse>()
				{
					@Override
					@Nullable
					public OntologyResponse apply(@Nullable Ontology ontology)
					{
						try
						{
							return ontology != null ? new OntologyResponse(ontology, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/ontology");
	}

	private static class OntologyRequest
	{
		private String identifier;
		private String name;
		private String ontologyAccession;
		private String ontologyURI;
	
		public Ontology toOntology()
		{
			Ontology ontology = new Ontology();
			ontology.setIdentifier(identifier);
			ontology.setName(name);
			ontology.setOntologyAccession(ontologyAccession);
			ontology.setOntologyURI(ontologyURI);
			return ontology;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setOntologyAccession(String ontologyAccession)
		{
			this.ontologyAccession = ontologyAccession;
		}
		
		public void setOntologyURI(String ontologyURI)
		{
			this.ontologyURI = ontologyURI;
		}
		
	}

	static class OntologyResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String ontologyAccession;
		private final String ontologyURI;
	
		public OntologyResponse(Ontology ontology, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/ontology/" + ontology.getId();
			this.identifier = ontology.getIdentifier();
			this.name = ontology.getName();
			this.ontologyAccession = ontology.getOntologyAccession();
			this.ontologyURI = ontology.getOntologyURI();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
		public String getOntologyAccession()
		{
			return ontologyAccession;
		}
	
		public String getOntologyURI()
		{
			return ontologyURI;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}