package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.CapturingKit;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.CapturingKitService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/capturingkit")
public class CapturingKitController
{
	@Autowired
	private CapturingKitService capturingKitService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<CapturingKitResponse> createCapturingKit(@Valid @RequestBody CapturingKitRequest capturingKitRequest)
			throws DatabaseException
	{
		return _createCapturingKit(capturingKitRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CapturingKitResponse> createCapturingKitFromForm(@Valid @ModelAttribute CapturingKitRequest capturingKitRequest)
			throws DatabaseException
	{
		return _createCapturingKit(capturingKitRequest);
	}

	private ResponseEntity<CapturingKitResponse> _createCapturingKit(CapturingKitRequest capturingKitRequest) throws DatabaseException
	{
		CapturingKit capturingKit = capturingKitService.create(capturingKitRequest.toCapturingKit());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/capturingkit/" + capturingKit.getId());
		return new ResponseEntity<CapturingKitResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public CapturingKitResponse retrieveCapturingKit(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCapturingKit(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public CapturingKitResponse retrieveCapturingKitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCapturingKit(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private CapturingKitResponse _retrieveCapturingKit(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		CapturingKit capturingKit = capturingKitService.read(id);
		if (capturingKit == null) throw new EntityNotFoundException("CapturingKit " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new CapturingKitResponse(capturingKit, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateCapturingKit(@PathVariable Integer id, @Valid @RequestBody CapturingKitRequest capturingKitRequest)
			throws DatabaseException
	{
		_updateCapturingKit(id, capturingKitRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CapturingKitResponse> updateCapturingKitFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute CapturingKitRequest capturingKitRequest) throws DatabaseException
	{
		return _createCapturingKit(capturingKitRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCapturingKitPost(@PathVariable Integer id, @Valid @RequestBody CapturingKitRequest capturingKitRequest)
			throws DatabaseException
	{
		_updateCapturingKit(id, capturingKitRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCapturingKitFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute CapturingKitRequest capturingKitRequest)
			throws DatabaseException
	{
		_updateCapturingKit(id, capturingKitRequest);
	}

	private void _updateCapturingKit(Integer id, CapturingKitRequest capturingKitRequest) throws DatabaseException
	{
		CapturingKit capturingKit = capturingKitRequest.toCapturingKit();
		capturingKit.setId(id);
		capturingKitService.update(capturingKit);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCapturingKit(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCapturingKit(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCapturingKitPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCapturingKit(id);
	}

	private void _deleteCapturingKit(Integer id) throws DatabaseException
	{
		boolean isDeleted = capturingKitService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("CapturingKit " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<CapturingKitResponse> retrieveCapturingKitCollection(@Valid EntityCollectionRequest capturingKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCapturingKitCollection(capturingKitCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CapturingKitResponse> retrieveCapturingKitCollectionJson(@Valid EntityCollectionRequest capturingKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCapturingKitCollection(capturingKitCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<CapturingKitResponse> retrieveCapturingKitCollectionPost(@Valid @RequestBody EntityCollectionRequest capturingKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCapturingKitCollection(capturingKitCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CapturingKitResponse> retrieveCapturingKitCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest capturingKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCapturingKitCollection(capturingKitCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<CapturingKitResponse> _retrieveCapturingKitCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<CapturingKit> capturingKitPager = capturingKitService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<CapturingKitResponse>(capturingKitPager, Lists.newArrayList(Iterables.transform(
				capturingKitPager.getIterable(), new Function<CapturingKit, CapturingKitResponse>()
				{
					@Override
					@Nullable
					public CapturingKitResponse apply(@Nullable CapturingKit capturingKit)
					{
						try
						{
							return capturingKit != null ? new CapturingKitResponse(capturingKit, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/capturingkit");
	}

	private static class CapturingKitRequest
	{
		private String capturingKitName;
	
		public CapturingKit toCapturingKit()
		{
			CapturingKit capturingKit = new CapturingKit();
			capturingKit.setCapturingKitName(capturingKitName);
			return capturingKit;
		}
		
		public void setCapturingKitName(String capturingKitName)
		{
			this.capturingKitName = capturingKitName;
		}
		
	}

	static class CapturingKitResponse
	{
		private final String href;
		private final String capturingKitName;
	
		public CapturingKitResponse(CapturingKit capturingKit, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/capturingkit/" + capturingKit.getId();
			this.capturingKitName = capturingKit.getCapturingKitName();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getCapturingKitName()
		{
			return capturingKitName;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}