package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.auth.Institute;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.InstituteService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/institute")
public class InstituteController
{
	@Autowired
	private InstituteService instituteService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<InstituteResponse> createInstitute(@Valid @RequestBody InstituteRequest instituteRequest)
			throws DatabaseException
	{
		return _createInstitute(instituteRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<InstituteResponse> createInstituteFromForm(@Valid @ModelAttribute InstituteRequest instituteRequest)
			throws DatabaseException
	{
		return _createInstitute(instituteRequest);
	}

	private ResponseEntity<InstituteResponse> _createInstitute(InstituteRequest instituteRequest) throws DatabaseException
	{
		Institute institute = instituteService.create(instituteRequest.toInstitute());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/institute/" + institute.getId());
		return new ResponseEntity<InstituteResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public InstituteResponse retrieveInstitute(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveInstitute(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public InstituteResponse retrieveInstituteJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveInstitute(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private InstituteResponse _retrieveInstitute(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Institute institute = instituteService.read(id);
		if (institute == null) throw new EntityNotFoundException("Institute " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new InstituteResponse(institute, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateInstitute(@PathVariable Integer id, @Valid @RequestBody InstituteRequest instituteRequest)
			throws DatabaseException
	{
		_updateInstitute(id, instituteRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<InstituteResponse> updateInstituteFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute InstituteRequest instituteRequest) throws DatabaseException
	{
		return _createInstitute(instituteRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateInstitutePost(@PathVariable Integer id, @Valid @RequestBody InstituteRequest instituteRequest)
			throws DatabaseException
	{
		_updateInstitute(id, instituteRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateInstituteFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute InstituteRequest instituteRequest)
			throws DatabaseException
	{
		_updateInstitute(id, instituteRequest);
	}

	private void _updateInstitute(Integer id, InstituteRequest instituteRequest) throws DatabaseException
	{
		Institute institute = instituteRequest.toInstitute();
		institute.setId(id);
		instituteService.update(institute);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteInstitute(@PathVariable Integer id) throws DatabaseException
	{
		_deleteInstitute(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteInstitutePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteInstitute(id);
	}

	private void _deleteInstitute(Integer id) throws DatabaseException
	{
		boolean isDeleted = instituteService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Institute " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<InstituteResponse> retrieveInstituteCollection(@Valid EntityCollectionRequest instituteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveInstituteCollection(instituteCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<InstituteResponse> retrieveInstituteCollectionJson(@Valid EntityCollectionRequest instituteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveInstituteCollection(instituteCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<InstituteResponse> retrieveInstituteCollectionPost(@Valid @RequestBody EntityCollectionRequest instituteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveInstituteCollection(instituteCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<InstituteResponse> retrieveInstituteCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest instituteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveInstituteCollection(instituteCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<InstituteResponse> _retrieveInstituteCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Institute> institutePager = instituteService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<InstituteResponse>(institutePager, Lists.newArrayList(Iterables.transform(
				institutePager.getIterable(), new Function<Institute, InstituteResponse>()
				{
					@Override
					@Nullable
					public InstituteResponse apply(@Nullable Institute institute)
					{
						try
						{
							return institute != null ? new InstituteResponse(institute, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/institute");
	}

	private static class InstituteRequest
	{
		private String identifier;
		private String name;
		private String description;
		private String address;
		private String phone;
		private String email;
		private String fax;
		private String tollFreePhone;
		private String city;
		private String country;
	
		public Institute toInstitute()
		{
			Institute institute = new Institute();
			institute.setIdentifier(identifier);
			institute.setName(name);
			institute.setDescription(description);
			institute.setAddress(address);
			institute.setPhone(phone);
			institute.setEmail(email);
			institute.setFax(fax);
			institute.setTollFreePhone(tollFreePhone);
			institute.setCity(city);
			institute.setCountry(country);
			return institute;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setAddress(String address)
		{
			this.address = address;
		}
		
		public void setPhone(String phone)
		{
			this.phone = phone;
		}
		
		public void setEmail(String email)
		{
			this.email = email;
		}
		
		public void setFax(String fax)
		{
			this.fax = fax;
		}
		
		public void setTollFreePhone(String tollFreePhone)
		{
			this.tollFreePhone = tollFreePhone;
		}
		
		public void setCity(String city)
		{
			this.city = city;
		}
		
		public void setCountry(String country)
		{
			this.country = country;
		}
		
	}

	static class InstituteResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final String address;
		private final String phone;
		private final String email;
		private final String fax;
		private final String tollFreePhone;
		private final String city;
		private final String country;
	
		public InstituteResponse(Institute institute, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/institute/" + institute.getId();
			this.identifier = institute.getIdentifier();
			this.name = institute.getName();
			this.description = institute.getDescription();
			this.address = institute.getAddress();
			this.phone = institute.getPhone();
			this.email = institute.getEmail();
			this.fax = institute.getFax();
			this.tollFreePhone = institute.getTollFreePhone();
			this.city = institute.getCity();
			this.country = institute.getCountry();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public String getAddress()
		{
			return address;
		}
	
		public String getPhone()
		{
			return phone;
		}
	
		public String getEmail()
		{
			return email;
		}
	
		public String getFax()
		{
			return fax;
		}
	
		public String getTollFreePhone()
		{
			return tollFreePhone;
		}
	
		public String getCity()
		{
			return city;
		}
	
		public String getCountry()
		{
			return country;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}