package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.NgsUser;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.NgsUserService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/ngsuser")
public class NgsUserController
{
	@Autowired
	private NgsUserService ngsUserService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<NgsUserResponse> createNgsUser(@Valid @RequestBody NgsUserRequest ngsUserRequest)
			throws DatabaseException
	{
		return _createNgsUser(ngsUserRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<NgsUserResponse> createNgsUserFromForm(@Valid @ModelAttribute NgsUserRequest ngsUserRequest)
			throws DatabaseException
	{
		return _createNgsUser(ngsUserRequest);
	}

	private ResponseEntity<NgsUserResponse> _createNgsUser(NgsUserRequest ngsUserRequest) throws DatabaseException
	{
		NgsUser ngsUser = ngsUserService.create(ngsUserRequest.toNgsUser());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/ngsuser/" + ngsUser.getId());
		return new ResponseEntity<NgsUserResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public NgsUserResponse retrieveNgsUser(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNgsUser(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public NgsUserResponse retrieveNgsUserJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNgsUser(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private NgsUserResponse _retrieveNgsUser(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		NgsUser ngsUser = ngsUserService.read(id);
		if (ngsUser == null) throw new EntityNotFoundException("NgsUser " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new NgsUserResponse(ngsUser, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateNgsUser(@PathVariable Integer id, @Valid @RequestBody NgsUserRequest ngsUserRequest)
			throws DatabaseException
	{
		_updateNgsUser(id, ngsUserRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<NgsUserResponse> updateNgsUserFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute NgsUserRequest ngsUserRequest) throws DatabaseException
	{
		return _createNgsUser(ngsUserRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateNgsUserPost(@PathVariable Integer id, @Valid @RequestBody NgsUserRequest ngsUserRequest)
			throws DatabaseException
	{
		_updateNgsUser(id, ngsUserRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateNgsUserFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute NgsUserRequest ngsUserRequest)
			throws DatabaseException
	{
		_updateNgsUser(id, ngsUserRequest);
	}

	private void _updateNgsUser(Integer id, NgsUserRequest ngsUserRequest) throws DatabaseException
	{
		NgsUser ngsUser = ngsUserRequest.toNgsUser();
		ngsUser.setId(id);
		ngsUserService.update(ngsUser);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteNgsUser(@PathVariable Integer id) throws DatabaseException
	{
		_deleteNgsUser(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteNgsUserPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteNgsUser(id);
	}

	private void _deleteNgsUser(Integer id) throws DatabaseException
	{
		boolean isDeleted = ngsUserService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("NgsUser " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<NgsUserResponse> retrieveNgsUserCollection(@Valid EntityCollectionRequest ngsUserCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNgsUserCollection(ngsUserCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<NgsUserResponse> retrieveNgsUserCollectionJson(@Valid EntityCollectionRequest ngsUserCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNgsUserCollection(ngsUserCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<NgsUserResponse> retrieveNgsUserCollectionPost(@Valid @RequestBody EntityCollectionRequest ngsUserCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNgsUserCollection(ngsUserCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<NgsUserResponse> retrieveNgsUserCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest ngsUserCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNgsUserCollection(ngsUserCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<NgsUserResponse> _retrieveNgsUserCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<NgsUser> ngsUserPager = ngsUserService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<NgsUserResponse>(ngsUserPager, Lists.newArrayList(Iterables.transform(
				ngsUserPager.getIterable(), new Function<NgsUser, NgsUserResponse>()
				{
					@Override
					@Nullable
					public NgsUserResponse apply(@Nullable NgsUser ngsUser)
					{
						try
						{
							return ngsUser != null ? new NgsUserResponse(ngsUser, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/ngsuser");
	}

	private static class NgsUserRequest
	{
		private String userName;
		private String userEmail;
		private String userRole;
		private String userGroup;
	
		public NgsUser toNgsUser()
		{
			NgsUser ngsUser = new NgsUser();
			ngsUser.setUserName(userName);
			ngsUser.setUserEmail(userEmail);
			ngsUser.setUserRole(userRole);
			ngsUser.setUserGroup(userGroup);
			return ngsUser;
		}
		
		public void setUserName(String userName)
		{
			this.userName = userName;
		}
		
		public void setUserEmail(String userEmail)
		{
			this.userEmail = userEmail;
		}
		
		public void setUserRole(String userRole)
		{
			this.userRole = userRole;
		}
		
		public void setUserGroup(String userGroup)
		{
			this.userGroup = userGroup;
		}
		
	}

	static class NgsUserResponse
	{
		private final String href;
		private final String userName;
		private final String userEmail;
		private final String userRole;
		private final String userGroup;
	
		public NgsUserResponse(NgsUser ngsUser, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/ngsuser/" + ngsUser.getId();
			this.userName = ngsUser.getUserName();
			this.userEmail = ngsUser.getUserEmail();
			this.userRole = ngsUser.getUserRole();
			this.userGroup = ngsUser.getUserGroup();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getUserName()
		{
			return userName;
		}
	
		public String getUserEmail()
		{
			return userEmail;
		}
	
		public String getUserRole()
		{
			return userRole;
		}
	
		public String getUserGroup()
		{
			return userGroup;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}