package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.target.Panel;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PanelService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.omx.observ.target.Species;
import org.molgenis.controller.SpeciesController.SpeciesResponse;
import org.molgenis.service.SpeciesService;
import org.molgenis.omx.observ.target.Individual;
import org.molgenis.controller.IndividualController.IndividualResponse;
import org.molgenis.service.IndividualService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/panel")
public class PanelController
{
	@Autowired
	private PanelService panelService;

	@Autowired
	private OntologyTermService ontologyTermService;
		
	@Autowired
	private SpeciesService speciesService;
		
	@Autowired
	private IndividualService individualService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PanelResponse> createPanel(@Valid @RequestBody PanelRequest panelRequest)
			throws DatabaseException
	{
		return _createPanel(panelRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PanelResponse> createPanelFromForm(@Valid @ModelAttribute PanelRequest panelRequest)
			throws DatabaseException
	{
		return _createPanel(panelRequest);
	}

	private ResponseEntity<PanelResponse> _createPanel(PanelRequest panelRequest) throws DatabaseException
	{
		Panel panel = panelService.create(panelRequest.toPanel());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/panel/" + panel.getId());
		return new ResponseEntity<PanelResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PanelResponse retrievePanel(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanel(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PanelResponse retrievePanelJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanel(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private PanelResponse _retrievePanel(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Panel panel = panelService.read(id);
		if (panel == null) throw new EntityNotFoundException("Panel " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PanelResponse(panel, expandFields);
	}
			
	@RequestMapping(value = "/{id}/panelType", method = RequestMethod.GET)
	public String retrievePanelXrefPanelType(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelXrefPanelType(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/panelType", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePanelXrefPanelTypeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelXrefPanelType(id, "json", expandFields);
	}
	
	private String _retrievePanelXrefPanelType(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Panel panel = panelService.read(id);
		if (panel == null) throw new EntityNotFoundException("Panel " + id.toString() + " not found");
		Integer ontologyTermId = panel.getPanelType_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/species", method = RequestMethod.GET)
	public String retrievePanelXrefSpecies(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelXrefSpecies(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/species", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePanelXrefSpeciesJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelXrefSpecies(id, "json", expandFields);
	}
	
	private String _retrievePanelXrefSpecies(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Panel panel = panelService.read(id);
		if (panel == null) throw new EntityNotFoundException("Panel " + id.toString() + " not found");
		Integer speciesId = panel.getSpecies_Id();
		String redirectUri = "redirect:/api/v1/species/" + speciesId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/individuals", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<IndividualResponse> retrievePanelMrefIndividuals(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Panel panel = panelService.read(id);
		if (panel == null) throw new EntityNotFoundException("Panel " + id.toString() + " not found");
		return _retrievePanelMrefIndividuals(panel, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/individuals", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<IndividualResponse> retrievePanelMrefIndividualsJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Panel panel = panelService.read(id);
		if (panel == null) throw new EntityNotFoundException("Panel " + id.toString() + " not found");
		return _retrievePanelMrefIndividuals(panel, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<IndividualResponse> _retrievePanelMrefIndividuals(Panel panel, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Individual> individualCollection = panel.getIndividuals();
		
		int total = individualCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		individualCollection = individualCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Individual> individualPager = new EntityPager<Individual>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, individualCollection);
		
		return new EntityCollectionResponse<IndividualResponse>(individualPager, Lists.newArrayList(Iterables.transform(individualCollection,
				new Function<Individual, IndividualResponse>()
				{
					@Override
					@Nullable
					public IndividualResponse apply(@Nullable Individual individual)
					{
						try
						{
							return individual != null ? new IndividualResponse(individual, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/panel/" + panel.getId() + "/individuals");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePanel(@PathVariable Integer id, @Valid @RequestBody PanelRequest panelRequest)
			throws DatabaseException
	{
		_updatePanel(id, panelRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PanelResponse> updatePanelFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PanelRequest panelRequest) throws DatabaseException
	{
		return _createPanel(panelRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePanelPost(@PathVariable Integer id, @Valid @RequestBody PanelRequest panelRequest)
			throws DatabaseException
	{
		_updatePanel(id, panelRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePanelFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PanelRequest panelRequest)
			throws DatabaseException
	{
		_updatePanel(id, panelRequest);
	}

	private void _updatePanel(Integer id, PanelRequest panelRequest) throws DatabaseException
	{
		Panel panel = panelRequest.toPanel();
		panel.setId(id);
		panelService.update(panel);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePanel(@PathVariable Integer id) throws DatabaseException
	{
		_deletePanel(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePanelPost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePanel(id);
	}

	private void _deletePanel(Integer id) throws DatabaseException
	{
		boolean isDeleted = panelService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Panel " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PanelResponse> retrievePanelCollection(@Valid EntityCollectionRequest panelCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelCollection(panelCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PanelResponse> retrievePanelCollectionJson(@Valid EntityCollectionRequest panelCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelCollection(panelCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PanelResponse> retrievePanelCollectionPost(@Valid @RequestBody EntityCollectionRequest panelCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelCollection(panelCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PanelResponse> retrievePanelCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest panelCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelCollection(panelCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PanelResponse> _retrievePanelCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Panel> panelPager = panelService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PanelResponse>(panelPager, Lists.newArrayList(Iterables.transform(
				panelPager.getIterable(), new Function<Panel, PanelResponse>()
				{
					@Override
					@Nullable
					public PanelResponse apply(@Nullable Panel panel)
					{
						try
						{
							return panel != null ? new PanelResponse(panel, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/panel");
	}

	private static class PanelRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer panelType;
		private Integer numberOfIndividuals;
		private Integer species;
		private java.util.List<Integer> individuals;
	
		public Panel toPanel()
		{
			Panel panel = new Panel();
			panel.setIdentifier(identifier);
			panel.setName(name);
			panel.setDescription(description);
			panel.setPanelType_Id(panelType);
			panel.setNumberOfIndividuals(numberOfIndividuals);
			panel.setSpecies_Id(species);
			panel.setIndividuals_Id(individuals);
			return panel;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setPanelType(Integer panelType)
		{
			this.panelType = panelType;
		}
		
		public void setNumberOfIndividuals(Integer numberOfIndividuals)
		{
			this.numberOfIndividuals = numberOfIndividuals;
		}
		
		public void setSpecies(Integer species)
		{
			this.species = species;
		}
		
		public void setIndividuals(java.util.List<Integer> individuals)
		{
			this.individuals = individuals;
		}
		
	}

	static class PanelResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object panelType;
		private final Integer numberOfIndividuals;
		private final Object species;
		private final Object individuals;
	
		public PanelResponse(Panel panel, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/panel/" + panel.getId();
			this.identifier = panel.getIdentifier();
			this.name = panel.getName();
			this.description = panel.getDescription();
			if (expandFields != null && expandFields.contains("panelType")) this.panelType = panel.getPanelType() == null ? null : new OntologyTermResponse(panel.getPanelType(), null);
			else this.panelType = panel.getPanelType() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/panel/" + panel.getId() + "/panelType");
			this.numberOfIndividuals = panel.getNumberOfIndividuals();
			if (expandFields != null && expandFields.contains("species")) this.species = panel.getSpecies() == null ? null : new SpeciesResponse(panel.getSpecies(), null);
			else this.species = panel.getSpecies() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/panel/" + panel.getId() + "/species");
			java.util.List<Individual> IndividualsCollection = panel.getIndividuals();
			if (expandFields != null && expandFields.contains("individuals")) this.individuals = IndividualsCollection == null ? null : _retrievePanelMrefIndividuals(panel, new EntityCollectionRequest());
			else this.individuals = IndividualsCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/panel/" + panel.getId() + "/individuals");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getPanelType()
		{
			return panelType;
		}
	
		public Integer getNumberOfIndividuals()
		{
			return numberOfIndividuals;
		}
	
		public Object getSpecies()
		{
			return species;
		}
	
		public Object getIndividuals()
		{
			return individuals;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}