package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.PrepKit;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PrepKitService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/prepkit")
public class PrepKitController
{
	@Autowired
	private PrepKitService prepKitService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PrepKitResponse> createPrepKit(@Valid @RequestBody PrepKitRequest prepKitRequest)
			throws DatabaseException
	{
		return _createPrepKit(prepKitRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PrepKitResponse> createPrepKitFromForm(@Valid @ModelAttribute PrepKitRequest prepKitRequest)
			throws DatabaseException
	{
		return _createPrepKit(prepKitRequest);
	}

	private ResponseEntity<PrepKitResponse> _createPrepKit(PrepKitRequest prepKitRequest) throws DatabaseException
	{
		PrepKit prepKit = prepKitService.create(prepKitRequest.toPrepKit());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/prepkit/" + prepKit.getId());
		return new ResponseEntity<PrepKitResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PrepKitResponse retrievePrepKit(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePrepKit(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PrepKitResponse retrievePrepKitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePrepKit(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private PrepKitResponse _retrievePrepKit(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		PrepKit prepKit = prepKitService.read(id);
		if (prepKit == null) throw new EntityNotFoundException("PrepKit " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PrepKitResponse(prepKit, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePrepKit(@PathVariable Integer id, @Valid @RequestBody PrepKitRequest prepKitRequest)
			throws DatabaseException
	{
		_updatePrepKit(id, prepKitRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PrepKitResponse> updatePrepKitFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PrepKitRequest prepKitRequest) throws DatabaseException
	{
		return _createPrepKit(prepKitRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePrepKitPost(@PathVariable Integer id, @Valid @RequestBody PrepKitRequest prepKitRequest)
			throws DatabaseException
	{
		_updatePrepKit(id, prepKitRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePrepKitFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PrepKitRequest prepKitRequest)
			throws DatabaseException
	{
		_updatePrepKit(id, prepKitRequest);
	}

	private void _updatePrepKit(Integer id, PrepKitRequest prepKitRequest) throws DatabaseException
	{
		PrepKit prepKit = prepKitRequest.toPrepKit();
		prepKit.setId(id);
		prepKitService.update(prepKit);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePrepKit(@PathVariable Integer id) throws DatabaseException
	{
		_deletePrepKit(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePrepKitPost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePrepKit(id);
	}

	private void _deletePrepKit(Integer id) throws DatabaseException
	{
		boolean isDeleted = prepKitService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("PrepKit " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PrepKitResponse> retrievePrepKitCollection(@Valid EntityCollectionRequest prepKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePrepKitCollection(prepKitCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PrepKitResponse> retrievePrepKitCollectionJson(@Valid EntityCollectionRequest prepKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePrepKitCollection(prepKitCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PrepKitResponse> retrievePrepKitCollectionPost(@Valid @RequestBody EntityCollectionRequest prepKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePrepKitCollection(prepKitCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PrepKitResponse> retrievePrepKitCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest prepKitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePrepKitCollection(prepKitCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PrepKitResponse> _retrievePrepKitCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<PrepKit> prepKitPager = prepKitService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PrepKitResponse>(prepKitPager, Lists.newArrayList(Iterables.transform(
				prepKitPager.getIterable(), new Function<PrepKit, PrepKitResponse>()
				{
					@Override
					@Nullable
					public PrepKitResponse apply(@Nullable PrepKit prepKit)
					{
						try
						{
							return prepKit != null ? new PrepKitResponse(prepKit, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/prepkit");
	}

	private static class PrepKitRequest
	{
		private String prepKitName;
	
		public PrepKit toPrepKit()
		{
			PrepKit prepKit = new PrepKit();
			prepKit.setPrepKitName(prepKitName);
			return prepKit;
		}
		
		public void setPrepKitName(String prepKitName)
		{
			this.prepKitName = prepKitName;
		}
		
	}

	static class PrepKitResponse
	{
		private final String href;
		private final String prepKitName;
	
		public PrepKitResponse(PrepKit prepKit, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/prepkit/" + prepKit.getId();
			this.prepKitName = prepKit.getPrepKitName();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getPrepKitName()
		{
			return prepKitName;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}