package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.Sample;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.SampleService;
import org.molgenis.omx.ngs.Project;
import org.molgenis.controller.ProjectController.ProjectResponse;
import org.molgenis.service.ProjectService;
import org.molgenis.omx.ngs.CapturingKit;
import org.molgenis.controller.CapturingKitController.CapturingKitResponse;
import org.molgenis.service.CapturingKitService;
import org.molgenis.omx.ngs.SampleBarcode;
import org.molgenis.controller.SampleBarcodeController.SampleBarcodeResponse;
import org.molgenis.service.SampleBarcodeService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/sample")
public class SampleController
{
	@Autowired
	private SampleService sampleService;

	@Autowired
	private ProjectService projectService;
		
	@Autowired
	private CapturingKitService capturingKitService;
		
	@Autowired
	private SampleBarcodeService sampleBarcodeService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<SampleResponse> createSample(@Valid @RequestBody SampleRequest sampleRequest)
			throws DatabaseException
	{
		return _createSample(sampleRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SampleResponse> createSampleFromForm(@Valid @ModelAttribute SampleRequest sampleRequest)
			throws DatabaseException
	{
		return _createSample(sampleRequest);
	}

	private ResponseEntity<SampleResponse> _createSample(SampleRequest sampleRequest) throws DatabaseException
	{
		Sample sample = sampleService.create(sampleRequest.toSample());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/sample/" + sample.getId());
		return new ResponseEntity<SampleResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public SampleResponse retrieveSample(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSample(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public SampleResponse retrieveSampleJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSample(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private SampleResponse _retrieveSample(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new SampleResponse(sample, expandFields);
	}
			
	@RequestMapping(value = "/{id}/projectId", method = RequestMethod.GET)
	public String retrieveSampleXrefProjectId(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefProjectId(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/projectId", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSampleXrefProjectIdJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefProjectId(id, "json", expandFields);
	}
	
	private String _retrieveSampleXrefProjectId(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Integer projectId = sample.getProjectId_Id();
		String redirectUri = "redirect:/api/v1/project/" + projectId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/capturingKit", method = RequestMethod.GET)
	public String retrieveSampleXrefCapturingKit(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefCapturingKit(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/capturingKit", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSampleXrefCapturingKitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefCapturingKit(id, "json", expandFields);
	}
	
	private String _retrieveSampleXrefCapturingKit(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Integer capturingKitId = sample.getCapturingKit_Id();
		String redirectUri = "redirect:/api/v1/capturingkit/" + capturingKitId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/sampleBarcode", method = RequestMethod.GET)
	public String retrieveSampleXrefSampleBarcode(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefSampleBarcode(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/sampleBarcode", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSampleXrefSampleBarcodeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefSampleBarcode(id, "json", expandFields);
	}
	
	private String _retrieveSampleXrefSampleBarcode(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Integer sampleBarcodeId = sample.getSampleBarcode_Id();
		String redirectUri = "redirect:/api/v1/samplebarcode/" + sampleBarcodeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/sampleInPool", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleMrefSampleInPool(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		return _retrieveSampleMrefSampleInPool(sample, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/sampleInPool", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleMrefSampleInPoolJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		return _retrieveSampleMrefSampleInPool(sample, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<SampleResponse> _retrieveSampleMrefSampleInPool(Sample sample, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Sample> sampleCollection = sample.getSampleInPool();
		
		int total = sampleCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		sampleCollection = sampleCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Sample> samplePager = new EntityPager<Sample>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, sampleCollection);
		
		return new EntityCollectionResponse<SampleResponse>(samplePager, Lists.newArrayList(Iterables.transform(sampleCollection,
				new Function<Sample, SampleResponse>()
				{
					@Override
					@Nullable
					public SampleResponse apply(@Nullable Sample sample)
					{
						try
						{
							return sample != null ? new SampleResponse(sample, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/sample/" + sample.getId() + "/sampleInPool");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateSample(@PathVariable Integer id, @Valid @RequestBody SampleRequest sampleRequest)
			throws DatabaseException
	{
		_updateSample(id, sampleRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SampleResponse> updateSampleFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute SampleRequest sampleRequest) throws DatabaseException
	{
		return _createSample(sampleRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSamplePost(@PathVariable Integer id, @Valid @RequestBody SampleRequest sampleRequest)
			throws DatabaseException
	{
		_updateSample(id, sampleRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSampleFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute SampleRequest sampleRequest)
			throws DatabaseException
	{
		_updateSample(id, sampleRequest);
	}

	private void _updateSample(Integer id, SampleRequest sampleRequest) throws DatabaseException
	{
		Sample sample = sampleRequest.toSample();
		sample.setId(id);
		sampleService.update(sample);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSample(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSample(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSamplePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSample(id);
	}

	private void _deleteSample(Integer id) throws DatabaseException
	{
		boolean isDeleted = sampleService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollection(@Valid EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollectionJson(@Valid EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollectionPost(@Valid @RequestBody EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<SampleResponse> _retrieveSampleCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Sample> samplePager = sampleService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<SampleResponse>(samplePager, Lists.newArrayList(Iterables.transform(
				samplePager.getIterable(), new Function<Sample, SampleResponse>()
				{
					@Override
					@Nullable
					public SampleResponse apply(@Nullable Sample sample)
					{
						try
						{
							return sample != null ? new SampleResponse(sample, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/sample");
	}

	private static class SampleRequest
	{
		private String internalId;
		private String externalId;
		private String sampleComment;
		private Integer projectId;
		private String arrayFile;
		private String arrayId;
		private Integer capturingKit;
		private Integer sampleBarcode;
		private java.util.List<Integer> sampleInPool;
		private String labStatus;
	
		public Sample toSample()
		{
			Sample sample = new Sample();
			sample.setInternalId(internalId);
			sample.setExternalId(externalId);
			sample.setSampleComment(sampleComment);
			sample.setProjectId_Id(projectId);
			sample.setArrayFile(arrayFile);
			sample.setArrayId(arrayId);
			sample.setCapturingKit_Id(capturingKit);
			sample.setSampleBarcode_Id(sampleBarcode);
			sample.setSampleInPool_Id(sampleInPool);
			sample.setLabStatus(labStatus);
			return sample;
		}
		
		public void setInternalId(String internalId)
		{
			this.internalId = internalId;
		}
		
		public void setExternalId(String externalId)
		{
			this.externalId = externalId;
		}
		
		public void setSampleComment(String sampleComment)
		{
			this.sampleComment = sampleComment;
		}
		
		public void setProjectId(Integer projectId)
		{
			this.projectId = projectId;
		}
		
		public void setArrayFile(String arrayFile)
		{
			this.arrayFile = arrayFile;
		}
		
		public void setArrayId(String arrayId)
		{
			this.arrayId = arrayId;
		}
		
		public void setCapturingKit(Integer capturingKit)
		{
			this.capturingKit = capturingKit;
		}
		
		public void setSampleBarcode(Integer sampleBarcode)
		{
			this.sampleBarcode = sampleBarcode;
		}
		
		public void setSampleInPool(java.util.List<Integer> sampleInPool)
		{
			this.sampleInPool = sampleInPool;
		}
		
		public void setLabStatus(String labStatus)
		{
			this.labStatus = labStatus;
		}
		
	}

	static class SampleResponse
	{
		private final String href;
		private final String internalId;
		private final String externalId;
		private final String sampleComment;
		private final Object projectId;
		private final String arrayFile;
		private final String arrayId;
		private final Object capturingKit;
		private final Object sampleBarcode;
		private final Object sampleInPool;
		private final String labStatus;
	
		public SampleResponse(Sample sample, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/sample/" + sample.getId();
			this.internalId = sample.getInternalId();
			this.externalId = sample.getExternalId();
			this.sampleComment = sample.getSampleComment();
			if (expandFields != null && expandFields.contains("projectId")) this.projectId = new ProjectResponse(sample.getProjectId(), null);
			else this.projectId = java.util.Collections.singletonMap("href", "/api/v1/sample/" + sample.getId() + "/projectId");
			this.arrayFile = sample.getArrayFile();
			this.arrayId = sample.getArrayId();
			if (expandFields != null && expandFields.contains("capturingKit")) this.capturingKit = sample.getCapturingKit() == null ? null : new CapturingKitResponse(sample.getCapturingKit(), null);
			else this.capturingKit = sample.getCapturingKit() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/sample/" + sample.getId() + "/capturingKit");
			if (expandFields != null && expandFields.contains("sampleBarcode")) this.sampleBarcode = sample.getSampleBarcode() == null ? null : new SampleBarcodeResponse(sample.getSampleBarcode(), null);
			else this.sampleBarcode = sample.getSampleBarcode() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/sample/" + sample.getId() + "/sampleBarcode");
			java.util.List<Sample> SampleInPoolCollection = sample.getSampleInPool();
			if (expandFields != null && expandFields.contains("sampleInPool")) this.sampleInPool = SampleInPoolCollection == null ? null : _retrieveSampleMrefSampleInPool(sample, new EntityCollectionRequest());
			else this.sampleInPool = SampleInPoolCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/sample/" + sample.getId() + "/sampleInPool");
			this.labStatus = sample.getLabStatus();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getInternalId()
		{
			return internalId;
		}
	
		public String getExternalId()
		{
			return externalId;
		}
	
		public String getSampleComment()
		{
			return sampleComment;
		}
	
		public Object getProjectId()
		{
			return projectId;
		}
	
		public String getArrayFile()
		{
			return arrayFile;
		}
	
		public String getArrayId()
		{
			return arrayId;
		}
	
		public Object getCapturingKit()
		{
			return capturingKit;
		}
	
		public Object getSampleBarcode()
		{
			return sampleBarcode;
		}
	
		public Object getSampleInPool()
		{
			return sampleInPool;
		}
	
		public String getLabStatus()
		{
			return labStatus;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}