package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.target.Species;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.SpeciesService;
import org.molgenis.omx.observ.target.Ontology;
import org.molgenis.controller.OntologyController.OntologyResponse;
import org.molgenis.service.OntologyService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/species")
public class SpeciesController
{
	@Autowired
	private SpeciesService speciesService;

	@Autowired
	private OntologyService ontologyService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<SpeciesResponse> createSpecies(@Valid @RequestBody SpeciesRequest speciesRequest)
			throws DatabaseException
	{
		return _createSpecies(speciesRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SpeciesResponse> createSpeciesFromForm(@Valid @ModelAttribute SpeciesRequest speciesRequest)
			throws DatabaseException
	{
		return _createSpecies(speciesRequest);
	}

	private ResponseEntity<SpeciesResponse> _createSpecies(SpeciesRequest speciesRequest) throws DatabaseException
	{
		Species species = speciesService.create(speciesRequest.toSpecies());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/species/" + species.getId());
		return new ResponseEntity<SpeciesResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public SpeciesResponse retrieveSpecies(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpecies(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public SpeciesResponse retrieveSpeciesJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpecies(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private SpeciesResponse _retrieveSpecies(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Species species = speciesService.read(id);
		if (species == null) throw new EntityNotFoundException("Species " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new SpeciesResponse(species, expandFields);
	}
			
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET)
	public String retrieveSpeciesXrefOntology(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpeciesXrefOntology(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSpeciesXrefOntologyJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpeciesXrefOntology(id, "json", expandFields);
	}
	
	private String _retrieveSpeciesXrefOntology(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Species species = speciesService.read(id);
		if (species == null) throw new EntityNotFoundException("Species " + id.toString() + " not found");
		Integer ontologyId = species.getOntology_Id();
		String redirectUri = "redirect:/api/v1/ontology/" + ontologyId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateSpecies(@PathVariable Integer id, @Valid @RequestBody SpeciesRequest speciesRequest)
			throws DatabaseException
	{
		_updateSpecies(id, speciesRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SpeciesResponse> updateSpeciesFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute SpeciesRequest speciesRequest) throws DatabaseException
	{
		return _createSpecies(speciesRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSpeciesPost(@PathVariable Integer id, @Valid @RequestBody SpeciesRequest speciesRequest)
			throws DatabaseException
	{
		_updateSpecies(id, speciesRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSpeciesFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute SpeciesRequest speciesRequest)
			throws DatabaseException
	{
		_updateSpecies(id, speciesRequest);
	}

	private void _updateSpecies(Integer id, SpeciesRequest speciesRequest) throws DatabaseException
	{
		Species species = speciesRequest.toSpecies();
		species.setId(id);
		speciesService.update(species);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSpecies(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSpecies(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSpeciesPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSpecies(id);
	}

	private void _deleteSpecies(Integer id) throws DatabaseException
	{
		boolean isDeleted = speciesService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Species " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<SpeciesResponse> retrieveSpeciesCollection(@Valid EntityCollectionRequest speciesCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpeciesCollection(speciesCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SpeciesResponse> retrieveSpeciesCollectionJson(@Valid EntityCollectionRequest speciesCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpeciesCollection(speciesCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<SpeciesResponse> retrieveSpeciesCollectionPost(@Valid @RequestBody EntityCollectionRequest speciesCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpeciesCollection(speciesCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SpeciesResponse> retrieveSpeciesCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest speciesCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpeciesCollection(speciesCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<SpeciesResponse> _retrieveSpeciesCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Species> speciesPager = speciesService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<SpeciesResponse>(speciesPager, Lists.newArrayList(Iterables.transform(
				speciesPager.getIterable(), new Function<Species, SpeciesResponse>()
				{
					@Override
					@Nullable
					public SpeciesResponse apply(@Nullable Species species)
					{
						try
						{
							return species != null ? new SpeciesResponse(species, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/species");
	}

	private static class SpeciesRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer ontology;
		private String termAccession;
		private String definition;
	
		public Species toSpecies()
		{
			Species species = new Species();
			species.setIdentifier(identifier);
			species.setName(name);
			species.setDescription(description);
			species.setOntology_Id(ontology);
			species.setTermAccession(termAccession);
			species.setDefinition(definition);
			return species;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setOntology(Integer ontology)
		{
			this.ontology = ontology;
		}
		
		public void setTermAccession(String termAccession)
		{
			this.termAccession = termAccession;
		}
		
		public void setDefinition(String definition)
		{
			this.definition = definition;
		}
		
	}

	static class SpeciesResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object ontology;
		private final String termAccession;
		private final String definition;
	
		public SpeciesResponse(Species species, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/species/" + species.getId();
			this.identifier = species.getIdentifier();
			this.name = species.getName();
			this.description = species.getDescription();
			if (expandFields != null && expandFields.contains("ontology")) this.ontology = species.getOntology() == null ? null : new OntologyResponse(species.getOntology(), null);
			else this.ontology = species.getOntology() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/species/" + species.getId() + "/ontology");
			this.termAccession = species.getTermAccession();
			this.definition = species.getDefinition();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getOntology()
		{
			return ontology;
		}
	
		public String getTermAccession()
		{
			return termAccession;
		}
	
		public String getDefinition()
		{
			return definition;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}