package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.Value;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/value")
public class ValueController
{
	@Autowired
	private ValueService valueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ValueResponse> createValue(@Valid @RequestBody ValueRequest valueRequest)
			throws DatabaseException
	{
		return _createValue(valueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ValueResponse> createValueFromForm(@Valid @ModelAttribute ValueRequest valueRequest)
			throws DatabaseException
	{
		return _createValue(valueRequest);
	}

	private ResponseEntity<ValueResponse> _createValue(ValueRequest valueRequest) throws DatabaseException
	{
		Value value = valueService.create(valueRequest.toValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/value/" + value.getId());
		return new ResponseEntity<ValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ValueResponse retrieveValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ValueResponse retrieveValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ValueResponse _retrieveValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Value value = valueService.read(id);
		if (value == null) throw new EntityNotFoundException("Value " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ValueResponse(value, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateValue(@PathVariable Integer id, @Valid @RequestBody ValueRequest valueRequest)
			throws DatabaseException
	{
		_updateValue(id, valueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ValueResponse> updateValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ValueRequest valueRequest) throws DatabaseException
	{
		return _createValue(valueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateValuePost(@PathVariable Integer id, @Valid @RequestBody ValueRequest valueRequest)
			throws DatabaseException
	{
		_updateValue(id, valueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ValueRequest valueRequest)
			throws DatabaseException
	{
		_updateValue(id, valueRequest);
	}

	private void _updateValue(Integer id, ValueRequest valueRequest) throws DatabaseException
	{
		Value value = valueRequest.toValue();
		value.setId(id);
		valueService.update(value);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteValue(id);
	}

	private void _deleteValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = valueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Value " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ValueResponse> retrieveValueCollection(@Valid EntityCollectionRequest valueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveValueCollection(valueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ValueResponse> retrieveValueCollectionJson(@Valid EntityCollectionRequest valueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveValueCollection(valueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ValueResponse> retrieveValueCollectionPost(@Valid @RequestBody EntityCollectionRequest valueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveValueCollection(valueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ValueResponse> retrieveValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest valueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveValueCollection(valueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ValueResponse> _retrieveValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Value> valuePager = valueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ValueResponse>(valuePager, Lists.newArrayList(Iterables.transform(
				valuePager.getIterable(), new Function<Value, ValueResponse>()
				{
					@Override
					@Nullable
					public ValueResponse apply(@Nullable Value value)
					{
						try
						{
							return value != null ? new ValueResponse(value, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/value");
	}

	private static class ValueRequest
	{
	
		public Value toValue()
		{
			Value value = new Value();
			return value;
		}
		
	}

	static class ValueResponse
	{
		private final String href;
	
		public ValueResponse(Value value, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/value/" + value.getId();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}