package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.Flowcell;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.FlowcellService;
import org.molgenis.omx.ngs.Machine;
import org.molgenis.controller.MachineController.MachineResponse;
import org.molgenis.service.MachineService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/flowcell")
public class FlowcellController
{
	@Autowired
	private FlowcellService flowcellService;

	@Autowired
	private MachineService machineService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<FlowcellResponse> createFlowcell(@Valid @RequestBody FlowcellRequest flowcellRequest)
			throws DatabaseException
	{
		return _createFlowcell(flowcellRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<FlowcellResponse> createFlowcellFromForm(@Valid @ModelAttribute FlowcellRequest flowcellRequest)
			throws DatabaseException
	{
		return _createFlowcell(flowcellRequest);
	}

	private ResponseEntity<FlowcellResponse> _createFlowcell(FlowcellRequest flowcellRequest) throws DatabaseException
	{
		Flowcell flowcell = flowcellService.create(flowcellRequest.toFlowcell());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/flowcell/" + flowcell.getId());
		return new ResponseEntity<FlowcellResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public FlowcellResponse retrieveFlowcell(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcell(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public FlowcellResponse retrieveFlowcellJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcell(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private FlowcellResponse _retrieveFlowcell(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Flowcell flowcell = flowcellService.read(id);
		if (flowcell == null) throw new EntityNotFoundException("Flowcell " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new FlowcellResponse(flowcell, expandFields);
	}
			
	@RequestMapping(value = "/{id}/machine", method = RequestMethod.GET)
	public String retrieveFlowcellXrefMachine(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcellXrefMachine(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/machine", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveFlowcellXrefMachineJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcellXrefMachine(id, "json", expandFields);
	}
	
	private String _retrieveFlowcellXrefMachine(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Flowcell flowcell = flowcellService.read(id);
		if (flowcell == null) throw new EntityNotFoundException("Flowcell " + id.toString() + " not found");
		Integer machineId = flowcell.getMachine_Id();
		String redirectUri = "redirect:/api/v1/machine/" + machineId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateFlowcell(@PathVariable Integer id, @Valid @RequestBody FlowcellRequest flowcellRequest)
			throws DatabaseException
	{
		_updateFlowcell(id, flowcellRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<FlowcellResponse> updateFlowcellFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute FlowcellRequest flowcellRequest) throws DatabaseException
	{
		return _createFlowcell(flowcellRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateFlowcellPost(@PathVariable Integer id, @Valid @RequestBody FlowcellRequest flowcellRequest)
			throws DatabaseException
	{
		_updateFlowcell(id, flowcellRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateFlowcellFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute FlowcellRequest flowcellRequest)
			throws DatabaseException
	{
		_updateFlowcell(id, flowcellRequest);
	}

	private void _updateFlowcell(Integer id, FlowcellRequest flowcellRequest) throws DatabaseException
	{
		Flowcell flowcell = flowcellRequest.toFlowcell();
		flowcell.setId(id);
		flowcellService.update(flowcell);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteFlowcell(@PathVariable Integer id) throws DatabaseException
	{
		_deleteFlowcell(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteFlowcellPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteFlowcell(id);
	}

	private void _deleteFlowcell(Integer id) throws DatabaseException
	{
		boolean isDeleted = flowcellService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Flowcell " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<FlowcellResponse> retrieveFlowcellCollection(@Valid EntityCollectionRequest flowcellCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcellCollection(flowcellCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<FlowcellResponse> retrieveFlowcellCollectionJson(@Valid EntityCollectionRequest flowcellCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcellCollection(flowcellCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<FlowcellResponse> retrieveFlowcellCollectionPost(@Valid @RequestBody EntityCollectionRequest flowcellCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcellCollection(flowcellCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<FlowcellResponse> retrieveFlowcellCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest flowcellCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveFlowcellCollection(flowcellCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<FlowcellResponse> _retrieveFlowcellCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Flowcell> flowcellPager = flowcellService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<FlowcellResponse>(flowcellPager, Lists.newArrayList(Iterables.transform(
				flowcellPager.getIterable(), new Function<Flowcell, FlowcellResponse>()
				{
					@Override
					@Nullable
					public FlowcellResponse apply(@Nullable Flowcell flowcell)
					{
						try
						{
							return flowcell != null ? new FlowcellResponse(flowcell, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/flowcell");
	}

	private static class FlowcellRequest
	{
		private String flowcellDirection;
		private String flowcellName;
		private Integer machine;
		private String run;
		private java.util.Date runDate;
	
		public Flowcell toFlowcell()
		{
			Flowcell flowcell = new Flowcell();
			flowcell.setFlowcellDirection(flowcellDirection);
			flowcell.setFlowcellName(flowcellName);
			flowcell.setMachine_Id(machine);
			flowcell.setRun(run);
			flowcell.setRunDate(runDate);
			return flowcell;
		}
		
		public void setFlowcellDirection(String flowcellDirection)
		{
			this.flowcellDirection = flowcellDirection;
		}
		
		public void setFlowcellName(String flowcellName)
		{
			this.flowcellName = flowcellName;
		}
		
		public void setMachine(Integer machine)
		{
			this.machine = machine;
		}
		
		public void setRun(String run)
		{
			this.run = run;
		}
		
		public void setRunDate(java.util.Date runDate)
		{
			this.runDate = runDate;
		}
		
	}

	static class FlowcellResponse
	{
		private final String href;
		private final String flowcellDirection;
		private final String flowcellName;
		private final Object machine;
		private final String run;
		private final java.util.Date runDate;
	
		public FlowcellResponse(Flowcell flowcell, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/flowcell/" + flowcell.getId();
			this.flowcellDirection = flowcell.getFlowcellDirection();
			this.flowcellName = flowcell.getFlowcellName();
			if (expandFields != null && expandFields.contains("machine")) this.machine = flowcell.getMachine() == null ? null : new MachineResponse(flowcell.getMachine(), null);
			else this.machine = flowcell.getMachine() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/flowcell/" + flowcell.getId() + "/machine");
			this.run = flowcell.getRun();
			this.runDate = flowcell.getRunDate();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getFlowcellDirection()
		{
			return flowcellDirection;
		}
	
		public String getFlowcellName()
		{
			return flowcellName;
		}
	
		public Object getMachine()
		{
			return machine;
		}
	
		public String getRun()
		{
			return run;
		}
	
		public java.util.Date getRunDate()
		{
			return runDate;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}