package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.ngs.Machine;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.MachineService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/machine")
public class MachineController
{
	@Autowired
	private MachineService machineService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<MachineResponse> createMachine(@Valid @RequestBody MachineRequest machineRequest)
			throws DatabaseException
	{
		return _createMachine(machineRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MachineResponse> createMachineFromForm(@Valid @ModelAttribute MachineRequest machineRequest)
			throws DatabaseException
	{
		return _createMachine(machineRequest);
	}

	private ResponseEntity<MachineResponse> _createMachine(MachineRequest machineRequest) throws DatabaseException
	{
		Machine machine = machineService.create(machineRequest.toMachine());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/machine/" + machine.getId());
		return new ResponseEntity<MachineResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public MachineResponse retrieveMachine(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMachine(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public MachineResponse retrieveMachineJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMachine(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private MachineResponse _retrieveMachine(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Machine machine = machineService.read(id);
		if (machine == null) throw new EntityNotFoundException("Machine " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new MachineResponse(machine, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateMachine(@PathVariable Integer id, @Valid @RequestBody MachineRequest machineRequest)
			throws DatabaseException
	{
		_updateMachine(id, machineRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MachineResponse> updateMachineFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute MachineRequest machineRequest) throws DatabaseException
	{
		return _createMachine(machineRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMachinePost(@PathVariable Integer id, @Valid @RequestBody MachineRequest machineRequest)
			throws DatabaseException
	{
		_updateMachine(id, machineRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMachineFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute MachineRequest machineRequest)
			throws DatabaseException
	{
		_updateMachine(id, machineRequest);
	}

	private void _updateMachine(Integer id, MachineRequest machineRequest) throws DatabaseException
	{
		Machine machine = machineRequest.toMachine();
		machine.setId(id);
		machineService.update(machine);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMachine(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMachine(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMachinePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMachine(id);
	}

	private void _deleteMachine(Integer id) throws DatabaseException
	{
		boolean isDeleted = machineService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Machine " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MachineResponse> retrieveMachineCollection(@Valid EntityCollectionRequest machineCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMachineCollection(machineCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MachineResponse> retrieveMachineCollectionJson(@Valid EntityCollectionRequest machineCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMachineCollection(machineCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<MachineResponse> retrieveMachineCollectionPost(@Valid @RequestBody EntityCollectionRequest machineCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMachineCollection(machineCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MachineResponse> retrieveMachineCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest machineCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMachineCollection(machineCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<MachineResponse> _retrieveMachineCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Machine> machinePager = machineService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<MachineResponse>(machinePager, Lists.newArrayList(Iterables.transform(
				machinePager.getIterable(), new Function<Machine, MachineResponse>()
				{
					@Override
					@Nullable
					public MachineResponse apply(@Nullable Machine machine)
					{
						try
						{
							return machine != null ? new MachineResponse(machine, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/machine");
	}

	private static class MachineRequest
	{
		private String machineName;
	
		public Machine toMachine()
		{
			Machine machine = new Machine();
			machine.setMachineName(machineName);
			return machine;
		}
		
		public void setMachineName(String machineName)
		{
			this.machineName = machineName;
		}
		
	}

	static class MachineResponse
	{
		private final String href;
		private final String machineName;
	
		public MachineResponse(Machine machine, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/machine/" + machine.getId();
			this.machineName = machine.getMachineName();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getMachineName()
		{
			return machineName;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}