package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.MrefValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.MrefValueService;
import org.molgenis.omx.observ.Characteristic;
import org.molgenis.controller.CharacteristicController.CharacteristicResponse;
import org.molgenis.service.CharacteristicService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/mrefvalue")
public class MrefValueController
{
	@Autowired
	private MrefValueService mrefValueService;

	@Autowired
	private CharacteristicService characteristicService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<MrefValueResponse> createMrefValue(@Valid @RequestBody MrefValueRequest mrefValueRequest)
			throws DatabaseException
	{
		return _createMrefValue(mrefValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MrefValueResponse> createMrefValueFromForm(@Valid @ModelAttribute MrefValueRequest mrefValueRequest)
			throws DatabaseException
	{
		return _createMrefValue(mrefValueRequest);
	}

	private ResponseEntity<MrefValueResponse> _createMrefValue(MrefValueRequest mrefValueRequest) throws DatabaseException
	{
		MrefValue mrefValue = mrefValueService.create(mrefValueRequest.toMrefValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/mrefvalue/" + mrefValue.getId());
		return new ResponseEntity<MrefValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public MrefValueResponse retrieveMrefValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMrefValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public MrefValueResponse retrieveMrefValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMrefValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private MrefValueResponse _retrieveMrefValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		MrefValue mrefValue = mrefValueService.read(id);
		if (mrefValue == null) throw new EntityNotFoundException("MrefValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new MrefValueResponse(mrefValue, expandFields);
	}
			

	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<CharacteristicResponse> retrieveMrefValueMrefValue(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		MrefValue mrefValue = mrefValueService.read(id);
		if (mrefValue == null) throw new EntityNotFoundException("MrefValue " + id.toString() + " not found");
		return _retrieveMrefValueMrefValue(mrefValue, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CharacteristicResponse> retrieveMrefValueMrefValueJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		MrefValue mrefValue = mrefValueService.read(id);
		if (mrefValue == null) throw new EntityNotFoundException("MrefValue " + id.toString() + " not found");
		return _retrieveMrefValueMrefValue(mrefValue, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<CharacteristicResponse> _retrieveMrefValueMrefValue(MrefValue mrefValue, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Characteristic> characteristicCollection = mrefValue.getValue();
		
		int total = characteristicCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		characteristicCollection = characteristicCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Characteristic> characteristicPager = new EntityPager<Characteristic>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, characteristicCollection);
		
		return new EntityCollectionResponse<CharacteristicResponse>(characteristicPager, Lists.newArrayList(Iterables.transform(characteristicCollection,
				new Function<Characteristic, CharacteristicResponse>()
				{
					@Override
					@Nullable
					public CharacteristicResponse apply(@Nullable Characteristic characteristic)
					{
						try
						{
							return characteristic != null ? new CharacteristicResponse(characteristic, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/mrefvalue/" + mrefValue.getId() + "/value");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateMrefValue(@PathVariable Integer id, @Valid @RequestBody MrefValueRequest mrefValueRequest)
			throws DatabaseException
	{
		_updateMrefValue(id, mrefValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MrefValueResponse> updateMrefValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute MrefValueRequest mrefValueRequest) throws DatabaseException
	{
		return _createMrefValue(mrefValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMrefValuePost(@PathVariable Integer id, @Valid @RequestBody MrefValueRequest mrefValueRequest)
			throws DatabaseException
	{
		_updateMrefValue(id, mrefValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMrefValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute MrefValueRequest mrefValueRequest)
			throws DatabaseException
	{
		_updateMrefValue(id, mrefValueRequest);
	}

	private void _updateMrefValue(Integer id, MrefValueRequest mrefValueRequest) throws DatabaseException
	{
		MrefValue mrefValue = mrefValueRequest.toMrefValue();
		mrefValue.setId(id);
		mrefValueService.update(mrefValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMrefValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMrefValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMrefValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMrefValue(id);
	}

	private void _deleteMrefValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = mrefValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("MrefValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MrefValueResponse> retrieveMrefValueCollection(@Valid EntityCollectionRequest mrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMrefValueCollection(mrefValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MrefValueResponse> retrieveMrefValueCollectionJson(@Valid EntityCollectionRequest mrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMrefValueCollection(mrefValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<MrefValueResponse> retrieveMrefValueCollectionPost(@Valid @RequestBody EntityCollectionRequest mrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMrefValueCollection(mrefValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MrefValueResponse> retrieveMrefValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest mrefValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMrefValueCollection(mrefValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<MrefValueResponse> _retrieveMrefValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<MrefValue> mrefValuePager = mrefValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<MrefValueResponse>(mrefValuePager, Lists.newArrayList(Iterables.transform(
				mrefValuePager.getIterable(), new Function<MrefValue, MrefValueResponse>()
				{
					@Override
					@Nullable
					public MrefValueResponse apply(@Nullable MrefValue mrefValue)
					{
						try
						{
							return mrefValue != null ? new MrefValueResponse(mrefValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/mrefvalue");
	}

	private static class MrefValueRequest
	{
		private java.util.List<Integer> value;
	
		public MrefValue toMrefValue()
		{
			MrefValue mrefValue = new MrefValue();
			mrefValue.setValue_Id(value);
			return mrefValue;
		}
		
		public void setValue(java.util.List<Integer> value)
		{
			this.value = value;
		}
		
	}

	static class MrefValueResponse
	{
		private final String href;
		private final Object value;
	
		public MrefValueResponse(MrefValue mrefValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/mrefvalue/" + mrefValue.getId();
			
			if (expandFields != null && expandFields.contains("value")) this.value = _retrieveMrefValueMrefValue(mrefValue, new EntityCollectionRequest());
			else this.value = java.util.Collections.singletonMap("href", "/api/v1/mrefvalue/" + mrefValue.getId() + "/value");
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public Object getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}