package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.ObservedValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ObservedValueService;
import org.molgenis.omx.observ.ObservationSet;
import org.molgenis.controller.ObservationSetController.ObservationSetResponse;
import org.molgenis.service.ObservationSetService;
import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.controller.ObservableFeatureController.ObservableFeatureResponse;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.omx.observ.value.Value;
import org.molgenis.controller.ValueController.ValueResponse;
import org.molgenis.service.ValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/observedvalue")
public class ObservedValueController
{
	@Autowired
	private ObservedValueService observedValueService;

	@Autowired
	private ObservationSetService observationSetService;
		
	@Autowired
	private ObservableFeatureService observableFeatureService;
		
	@Autowired
	private ValueService valueService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ObservedValueResponse> createObservedValue(@Valid @RequestBody ObservedValueRequest observedValueRequest)
			throws DatabaseException
	{
		return _createObservedValue(observedValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservedValueResponse> createObservedValueFromForm(@Valid @ModelAttribute ObservedValueRequest observedValueRequest)
			throws DatabaseException
	{
		return _createObservedValue(observedValueRequest);
	}

	private ResponseEntity<ObservedValueResponse> _createObservedValue(ObservedValueRequest observedValueRequest) throws DatabaseException
	{
		ObservedValue observedValue = observedValueService.create(observedValueRequest.toObservedValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/observedvalue/" + observedValue.getId());
		return new ResponseEntity<ObservedValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ObservedValueResponse retrieveObservedValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ObservedValueResponse retrieveObservedValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ObservedValueResponse _retrieveObservedValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ObservedValue observedValue = observedValueService.read(id);
		if (observedValue == null) throw new EntityNotFoundException("ObservedValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ObservedValueResponse(observedValue, expandFields);
	}
			
	@RequestMapping(value = "/{id}/observationSet", method = RequestMethod.GET)
	public String retrieveObservedValueXrefObservationSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueXrefObservationSet(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/observationSet", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservedValueXrefObservationSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueXrefObservationSet(id, "json", expandFields);
	}
	
	private String _retrieveObservedValueXrefObservationSet(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservedValue observedValue = observedValueService.read(id);
		if (observedValue == null) throw new EntityNotFoundException("ObservedValue " + id.toString() + " not found");
		Integer observationSetId = observedValue.getObservationSet_Id();
		String redirectUri = "redirect:/api/v1/observationset/" + observationSetId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/feature", method = RequestMethod.GET)
	public String retrieveObservedValueXrefFeature(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueXrefFeature(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/feature", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservedValueXrefFeatureJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueXrefFeature(id, "json", expandFields);
	}
	
	private String _retrieveObservedValueXrefFeature(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservedValue observedValue = observedValueService.read(id);
		if (observedValue == null) throw new EntityNotFoundException("ObservedValue " + id.toString() + " not found");
		Integer observableFeatureId = observedValue.getFeature_Id();
		String redirectUri = "redirect:/api/v1/observablefeature/" + observableFeatureId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET)
	public String retrieveObservedValueXrefValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueXrefValue(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservedValueXrefValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueXrefValue(id, "json", expandFields);
	}
	
	private String _retrieveObservedValueXrefValue(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservedValue observedValue = observedValueService.read(id);
		if (observedValue == null) throw new EntityNotFoundException("ObservedValue " + id.toString() + " not found");
		Integer valueId = observedValue.getValue_Id();
		String redirectUri = "redirect:/api/v1/value/" + valueId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateObservedValue(@PathVariable Integer id, @Valid @RequestBody ObservedValueRequest observedValueRequest)
			throws DatabaseException
	{
		_updateObservedValue(id, observedValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservedValueResponse> updateObservedValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ObservedValueRequest observedValueRequest) throws DatabaseException
	{
		return _createObservedValue(observedValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservedValuePost(@PathVariable Integer id, @Valid @RequestBody ObservedValueRequest observedValueRequest)
			throws DatabaseException
	{
		_updateObservedValue(id, observedValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservedValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ObservedValueRequest observedValueRequest)
			throws DatabaseException
	{
		_updateObservedValue(id, observedValueRequest);
	}

	private void _updateObservedValue(Integer id, ObservedValueRequest observedValueRequest) throws DatabaseException
	{
		ObservedValue observedValue = observedValueRequest.toObservedValue();
		observedValue.setId(id);
		observedValueService.update(observedValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservedValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservedValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservedValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservedValue(id);
	}

	private void _deleteObservedValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = observedValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ObservedValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservedValueResponse> retrieveObservedValueCollection(@Valid EntityCollectionRequest observedValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueCollection(observedValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservedValueResponse> retrieveObservedValueCollectionJson(@Valid EntityCollectionRequest observedValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueCollection(observedValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ObservedValueResponse> retrieveObservedValueCollectionPost(@Valid @RequestBody EntityCollectionRequest observedValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueCollection(observedValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservedValueResponse> retrieveObservedValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest observedValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservedValueCollection(observedValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ObservedValueResponse> _retrieveObservedValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ObservedValue> observedValuePager = observedValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ObservedValueResponse>(observedValuePager, Lists.newArrayList(Iterables.transform(
				observedValuePager.getIterable(), new Function<ObservedValue, ObservedValueResponse>()
				{
					@Override
					@Nullable
					public ObservedValueResponse apply(@Nullable ObservedValue observedValue)
					{
						try
						{
							return observedValue != null ? new ObservedValueResponse(observedValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/observedvalue");
	}

	private static class ObservedValueRequest
	{
		private Integer observationSet;
		private Integer feature;
		private Integer value;
	
		public ObservedValue toObservedValue()
		{
			ObservedValue observedValue = new ObservedValue();
			observedValue.setObservationSet_Id(observationSet);
			observedValue.setFeature_Id(feature);
			observedValue.setValue_Id(value);
			return observedValue;
		}
		
		public void setObservationSet(Integer observationSet)
		{
			this.observationSet = observationSet;
		}
		
		public void setFeature(Integer feature)
		{
			this.feature = feature;
		}
		
		public void setValue(Integer value)
		{
			this.value = value;
		}
		
	}

	static class ObservedValueResponse
	{
		private final String href;
		private final Object observationSet;
		private final Object feature;
		private final Object value;
	
		public ObservedValueResponse(ObservedValue observedValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/observedvalue/" + observedValue.getId();
			if (expandFields != null && expandFields.contains("observationSet")) this.observationSet = new ObservationSetResponse(observedValue.getObservationSet(), null);
			else this.observationSet = java.util.Collections.singletonMap("href", "/api/v1/observedvalue/" + observedValue.getId() + "/observationSet");
			if (expandFields != null && expandFields.contains("feature")) this.feature = new ObservableFeatureResponse(observedValue.getFeature(), null);
			else this.feature = java.util.Collections.singletonMap("href", "/api/v1/observedvalue/" + observedValue.getId() + "/feature");
			if (expandFields != null && expandFields.contains("value")) this.value = observedValue.getValue() == null ? null : new ValueResponse(observedValue.getValue(), null);
			else this.value = observedValue.getValue() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/observedvalue/" + observedValue.getId() + "/value");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public Object getObservationSet()
		{
			return observationSet;
		}
	
		public Object getFeature()
		{
			return feature;
		}
	
		public Object getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}