package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.auth.PersonRole;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PersonRoleService;
import org.molgenis.omx.observ.target.Ontology;
import org.molgenis.controller.OntologyController.OntologyResponse;
import org.molgenis.service.OntologyService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/personrole")
public class PersonRoleController
{
	@Autowired
	private PersonRoleService personRoleService;

	@Autowired
	private OntologyService ontologyService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PersonRoleResponse> createPersonRole(@Valid @RequestBody PersonRoleRequest personRoleRequest)
			throws DatabaseException
	{
		return _createPersonRole(personRoleRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PersonRoleResponse> createPersonRoleFromForm(@Valid @ModelAttribute PersonRoleRequest personRoleRequest)
			throws DatabaseException
	{
		return _createPersonRole(personRoleRequest);
	}

	private ResponseEntity<PersonRoleResponse> _createPersonRole(PersonRoleRequest personRoleRequest) throws DatabaseException
	{
		PersonRole personRole = personRoleService.create(personRoleRequest.toPersonRole());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/personrole/" + personRole.getId());
		return new ResponseEntity<PersonRoleResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PersonRoleResponse retrievePersonRole(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRole(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PersonRoleResponse retrievePersonRoleJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRole(id, expandFields);
	}

	private PersonRoleResponse _retrievePersonRole(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		PersonRole personRole = personRoleService.read(id);
		if (personRole == null) throw new EntityNotFoundException("PersonRole " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PersonRoleResponse(personRole, expandFields);
	}
			
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET)
	public String retrievePersonRoleXrefOntology(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRoleXrefOntology(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePersonRoleXrefOntologyJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRoleXrefOntology(id, "json", expandFields);
	}
	
	private String _retrievePersonRoleXrefOntology(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		PersonRole personRole = personRoleService.read(id);
		if (personRole == null) throw new EntityNotFoundException("PersonRole " + id.toString() + " not found");
		Integer ontologyId = personRole.getOntology_Id();
		String redirectUri = "redirect:/api/v1/ontology/" + ontologyId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePersonRole(@PathVariable Integer id, @Valid @RequestBody PersonRoleRequest personRoleRequest)
			throws DatabaseException
	{
		_updatePersonRole(id, personRoleRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PersonRoleResponse> updatePersonRoleFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PersonRoleRequest personRoleRequest) throws DatabaseException
	{
		return _createPersonRole(personRoleRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePersonRolePost(@PathVariable Integer id, @Valid @RequestBody PersonRoleRequest personRoleRequest)
			throws DatabaseException
	{
		_updatePersonRole(id, personRoleRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePersonRoleFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PersonRoleRequest personRoleRequest)
			throws DatabaseException
	{
		_updatePersonRole(id, personRoleRequest);
	}

	private void _updatePersonRole(Integer id, PersonRoleRequest personRoleRequest) throws DatabaseException
	{
		PersonRole personRole = personRoleRequest.toPersonRole();
		personRole.setId(id);
		personRoleService.update(personRole);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePersonRole(@PathVariable Integer id) throws DatabaseException
	{
		_deletePersonRole(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePersonRolePost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePersonRole(id);
	}

	private void _deletePersonRole(Integer id) throws DatabaseException
	{
		boolean isDeleted = personRoleService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("PersonRole " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PersonRoleResponse> retrievePersonRoleCollection(@Valid EntityCollectionRequest personRoleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRoleCollection(personRoleCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PersonRoleResponse> retrievePersonRoleCollectionJson(@Valid EntityCollectionRequest personRoleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRoleCollection(personRoleCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PersonRoleResponse> retrievePersonRoleCollectionPost(@Valid @RequestBody EntityCollectionRequest personRoleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRoleCollection(personRoleCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PersonRoleResponse> retrievePersonRoleCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest personRoleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonRoleCollection(personRoleCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PersonRoleResponse> _retrievePersonRoleCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<PersonRole> personRolePager = personRoleService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PersonRoleResponse>(personRolePager, Lists.newArrayList(Iterables.transform(
				personRolePager.getIterable(), new Function<PersonRole, PersonRoleResponse>()
				{
					@Override
					@Nullable
					public PersonRoleResponse apply(@Nullable PersonRole personRole)
					{
						try
						{
							return personRole != null ? new PersonRoleResponse(personRole, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/personrole");
	}

	private static class PersonRoleRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer ontology;
		private String termAccession;
		private String definition;
	
		public PersonRole toPersonRole()
		{
			PersonRole personRole = new PersonRole();
			personRole.setIdentifier(identifier);
			personRole.setName(name);
			personRole.setDescription(description);
			personRole.setOntology_Id(ontology);
			personRole.setTermAccession(termAccession);
			personRole.setDefinition(definition);
			return personRole;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setOntology(Integer ontology)
		{
			this.ontology = ontology;
		}
		
		public void setTermAccession(String termAccession)
		{
			this.termAccession = termAccession;
		}
		
		public void setDefinition(String definition)
		{
			this.definition = definition;
		}
		
	}

	static class PersonRoleResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object ontology;
		private final String termAccession;
		private final String definition;
	
		public PersonRoleResponse(PersonRole personRole, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/personrole/" + personRole.getId();
			this.identifier = personRole.getIdentifier();
			this.name = personRole.getName();
			this.description = personRole.getDescription();
			if (expandFields != null && expandFields.contains("ontology")) this.ontology = personRole.getOntology() == null ? null : new OntologyResponse(personRole.getOntology(), null);
			else this.ontology = personRole.getOntology() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/personrole/" + personRole.getId() + "/ontology");
			this.termAccession = personRole.getTermAccession();
			this.definition = personRole.getDefinition();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getOntology()
		{
			return ontology;
		}
	
		public String getTermAccession()
		{
			return termAccession;
		}
	
		public String getDefinition()
		{
			return definition;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}